<?php

require_once 'vendor/autoload.php';

use App\Services\KenyaTaxCalculator;
use App\Services\PayrollService;

echo "=== QAYA HR PAYROLL SYSTEM TEST ===\n\n";

// Test 1: Tax Calculator Preview
echo "1. Testing Tax Calculator for different employment types:\n";
echo "---------------------------------------------------\n";

$testCases = [
    [
        'employment_type' => 'permanent',
        'basic_salary' => 80000,
        'house_allowance' => 15000,
        'transport_allowance' => 5000,
        'medical_allowance' => 3000,
    ],
    [
        'employment_type' => 'contract',
        'basic_salary' => 60000,
        'house_allowance' => 10000,
        'transport_allowance' => 3000,
        'statutory_deductions_applicable' => false,
        'withholding_tax_rate' => 0.05,
    ],
    [
        'employment_type' => 'casual',
        'basic_salary' => 25000,
        'statutory_deductions_applicable' => false,
    ],
    [
        'employment_type' => 'intern',
        'basic_salary' => 15000,
    ],
];

foreach ($testCases as $i => $testCase) {
    echo "\nTest Case " . ($i + 1) . " - " . ucfirst($testCase['employment_type']) . " Employee:\n";
    
    $calculations = KenyaTaxCalculator::calculatePayrollByEmploymentType(
        $testCase, 
        $testCase['employment_type']
    );
    
    echo "  Gross Salary: KES " . number_format($calculations['gross_salary'], 2) . "\n";
    echo "  PAYE Tax: KES " . number_format($calculations['paye_tax'], 2) . "\n";
    echo "  NHIF: KES " . number_format($calculations['nhif_deduction'], 2) . "\n";
    echo "  NSSF: KES " . number_format($calculations['nssf_employee'], 2) . "\n";
    echo "  Housing Levy: KES " . number_format($calculations['housing_levy'], 2) . "\n";
    
    if (isset($calculations['withholding_tax'])) {
        echo "  Withholding Tax: KES " . number_format($calculations['withholding_tax'], 2) . "\n";
    }
    
    echo "  Total Deductions: KES " . number_format($calculations['total_deductions'], 2) . "\n";
    echo "  Net Salary: KES " . number_format($calculations['net_salary'], 2) . "\n";
}

// Test 2: Hourly and Daily Calculations
echo "\n\n2. Testing Hourly and Daily Calculations:\n";
echo "----------------------------------------\n";

// Hourly calculation
$hourlyCalc = KenyaTaxCalculator::calculateHourlyPayroll(
    1500, // hourly rate
    160,  // regular hours
    20,   // overtime hours
    1.5,  // overtime multiplier
    ['house_allowance' => 5000, 'transport_allowance' => 2000]
);

echo "\nHourly Worker (160 regular + 20 overtime hours @ KES 1,500/hr):\n";
echo "  Basic Pay: KES " . number_format($hourlyCalc['basic_salary'], 2) . "\n";
echo "  Overtime Pay: KES " . number_format($hourlyCalc['overtime_pay'], 2) . "\n";
echo "  Gross Salary: KES " . number_format($hourlyCalc['gross_salary'], 2) . "\n";
echo "  Net Salary: KES " . number_format($hourlyCalc['net_salary'], 2) . "\n";

// Daily calculation
$dailyCalc = KenyaTaxCalculator::calculateDailyPayroll(
    3000, // daily rate
    22,   // days worked
    ['house_allowance' => 8000, 'transport_allowance' => 3000]
);

echo "\nDaily Worker (22 days @ KES 3,000/day):\n";
echo "  Basic Pay: KES " . number_format($dailyCalc['basic_salary'], 2) . "\n";
echo "  Gross Salary: KES " . number_format($dailyCalc['gross_salary'], 2) . "\n";
echo "  Net Salary: KES " . number_format($dailyCalc['net_salary'], 2) . "\n";

// Test 3: Employment Type Tax Rules
echo "\n\n3. Employment Type Tax Rules:\n";
echo "----------------------------\n";

$employmentTypes = ['permanent', 'contract', 'freelance', 'casual', 'intern', 'part_time'];

foreach ($employmentTypes as $type) {
    $rules = KenyaTaxCalculator::getEmploymentTypeTaxRules($type);
    echo "\n" . ucfirst($type) . " Employee:\n";
    echo "  PAYE: " . ($rules['paye_applicable'] ? 'Yes' : 'No') . "\n";
    echo "  NHIF: " . ($rules['nhif_applicable'] ? 'Yes' : 'No') . "\n";
    echo "  NSSF: " . ($rules['nssf_applicable'] ? 'Yes' : 'No') . "\n";
    echo "  Housing Levy: " . ($rules['housing_levy_applicable'] ? 'Yes' : 'No') . "\n";
    echo "  Withholding Tax: " . ($rules['withholding_tax_applicable'] ? 'Yes (' . ($rules['withholding_tax_rate'] * 100) . '%)' : 'No') . "\n";
}

echo "\n\n=== PAYROLL SYSTEM FEATURES SUMMARY ===\n";
echo "✓ Multi-employment type support (Permanent, Contract, Freelance, Casual, Intern, Part-time)\n";
echo "✓ Kenya-compliant tax calculations (PAYE, NHIF, NSSF, Housing Levy)\n";
echo "✓ Withholding tax for contractors and freelancers\n";
echo "✓ Flexible payment structures (Monthly, Daily, Hourly)\n";
echo "✓ Overtime calculations with configurable multipliers\n";
echo "✓ Attendance-based payroll generation\n";
echo "✓ Proration for partial attendance\n";
echo "✓ Comprehensive allowances and deductions\n";
echo "✓ API endpoints for integration\n";
echo "✓ Detailed calculation breakdowns\n";
echo "✓ Employment type-specific tax rules\n";

echo "\n=== API ENDPOINTS AVAILABLE ===\n";
echo "POST /api/payroll/preview - Preview payroll calculations\n";
echo "POST /api/payroll/generate-monthly - Generate payroll for all employees\n";
echo "POST /api/payroll/generate-employee/{id} - Generate payroll for specific employee\n";
echo "GET /api/payroll/summary - Get payroll summary for a period\n";
echo "GET /api/payroll/tax-rules - Get employment type tax rules\n";
echo "POST /api/payroll/calculate-hourly - Calculate hourly-based payroll\n";
echo "POST /api/payroll/calculate-daily - Calculate daily-based payroll\n";
echo "GET /api/payroll - List all payrolls with filters\n";
echo "PUT /api/payroll/{id} - Update payroll\n";
echo "POST /api/payroll/{id}/approve - Approve payroll\n";
echo "POST /api/payroll/{id}/process - Process approved payroll\n";

echo "\n=== TEST COMPLETED SUCCESSFULLY ===\n";