#!/usr/bin/env php
<?php

/**
 * Email Setup Verification Script
 * 
 * This script verifies that email notifications are properly configured.
 * Run this to check if emails will be sent correctly.
 * 
 * Usage: php test_email_setup.php
 */

echo "📧 Email Setup Verification\n";
echo "===========================\n\n";

$checks = [
    'passed' => [],
    'failed' => [],
    'warnings' => [],
];

// Check 1: Mailable classes exist
echo "Checking Mailable Classes...\n";
$mailables = [
    'app/Mail/OrganizationAdminCreated.php',
    'app/Mail/EmployeeAccountCreated.php',
];

foreach ($mailables as $mailable) {
    if (file_exists($mailable)) {
        $checks['passed'][] = "✅ {$mailable} exists";
    } else {
        $checks['failed'][] = "❌ {$mailable} NOT FOUND";
    }
}

// Check 2: Email views exist
echo "\nChecking Email Views...\n";
$views = [
    'resources/views/emails/organization-admin-created.blade.php',
    'resources/views/emails/employee-account-created.blade.php',
];

foreach ($views as $view) {
    if (file_exists($view)) {
        $checks['passed'][] = "✅ {$view} exists";
        
        // Check if view has required variables
        $content = file_get_contents($view);
        if (strpos($view, 'organization-admin') !== false) {
            if (strpos($content, '$organization') !== false && 
                strpos($content, '$adminUser') !== false && 
                strpos($content, '$password') !== false) {
                $checks['passed'][] = "✅ Organization admin view has required variables";
            } else {
                $checks['warnings'][] = "⚠️  Organization admin view may be missing variables";
            }
        } else {
            if (strpos($content, '$employee') !== false && 
                strpos($content, '$user') !== false && 
                strpos($content, '$password') !== false) {
                $checks['passed'][] = "✅ Employee view has required variables";
            } else {
                $checks['warnings'][] = "⚠️  Employee view may be missing variables";
            }
        }
    } else {
        $checks['failed'][] = "❌ {$view} NOT FOUND";
    }
}

// Check 3: Controllers have email sending code
echo "\nChecking Controllers...\n";
$controllers = [
    'app/Http/Controllers/OrganizationController.php' => 'sendSystemMail',
    'app/Http/Controllers/EmployeeController.php' => 'sendOrganizationMail',
];

foreach ($controllers as $controller => $method) {
    if (file_exists($controller)) {
        $content = file_get_contents($controller);
        if (strpos($content, $method) !== false) {
            $checks['passed'][] = "✅ {$controller} has {$method}() call";
        } else {
            $checks['failed'][] = "❌ {$controller} missing {$method}() call";
        }
        
        if (strpos($content, 'use App\Traits\SendsOrganizationMail') !== false) {
            $checks['passed'][] = "✅ {$controller} uses SendsOrganizationMail trait";
        } else {
            $checks['failed'][] = "❌ {$controller} missing SendsOrganizationMail trait";
        }
    } else {
        $checks['failed'][] = "❌ {$controller} NOT FOUND";
    }
}

// Check 4: Trait exists
echo "\nChecking Trait...\n";
$trait = 'app/Traits/SendsOrganizationMail.php';
if (file_exists($trait)) {
    $checks['passed'][] = "✅ SendsOrganizationMail trait exists";
    
    $content = file_get_contents($trait);
    if (strpos($content, 'sendOrganizationMail') !== false) {
        $checks['passed'][] = "✅ Trait has sendOrganizationMail() method";
    } else {
        $checks['failed'][] = "❌ Trait missing sendOrganizationMail() method";
    }
    
    if (strpos($content, 'sendSystemMail') !== false) {
        $checks['passed'][] = "✅ Trait has sendSystemMail() method";
    } else {
        $checks['failed'][] = "❌ Trait missing sendSystemMail() method";
    }
} else {
    $checks['failed'][] = "❌ SendsOrganizationMail trait NOT FOUND";
}

// Check 5: Service exists
echo "\nChecking Service...\n";
$service = 'app/Services/OrganizationMailService.php';
if (file_exists($service)) {
    $checks['passed'][] = "✅ OrganizationMailService exists";
    
    $content = file_get_contents($service);
    if (strpos($content, 'configureForOrganization') !== false) {
        $checks['passed'][] = "✅ Service has configureForOrganization() method";
    } else {
        $checks['failed'][] = "❌ Service missing configureForOrganization() method";
    }
} else {
    $checks['failed'][] = "❌ OrganizationMailService NOT FOUND";
}

// Check 6: .env file has mail configuration
echo "\nChecking .env Configuration...\n";
if (file_exists('.env')) {
    $envContent = file_get_contents('.env');
    
    $requiredVars = [
        'MAIL_MAILER',
        'MAIL_HOST',
        'MAIL_PORT',
        'MAIL_USERNAME',
        'MAIL_FROM_ADDRESS',
    ];
    
    foreach ($requiredVars as $var) {
        if (preg_match("/{$var}=(.+)/", $envContent, $matches)) {
            $value = trim($matches[1]);
            if (!empty($value) && $value !== 'null' && $value !== '""') {
                $checks['passed'][] = "✅ {$var} is configured";
            } else {
                $checks['warnings'][] = "⚠️  {$var} is empty or null";
            }
        } else {
            $checks['warnings'][] = "⚠️  {$var} not found in .env";
        }
    }
} else {
    $checks['failed'][] = "❌ .env file NOT FOUND";
}

// Check 7: Organization model has email fields
echo "\nChecking Organization Model...\n";
$orgModel = 'app/Models/Organization.php';
if (file_exists($orgModel)) {
    $content = file_get_contents($orgModel);
    $emailFields = [
        'use_custom_mail',
        'mail_host',
        'mail_port',
        'mail_username',
        'mail_from_address',
    ];
    
    $allFound = true;
    foreach ($emailFields as $field) {
        if (strpos($content, "'{$field}'") === false) {
            $allFound = false;
            break;
        }
    }
    
    if ($allFound) {
        $checks['passed'][] = "✅ Organization model has email configuration fields";
    } else {
        $checks['warnings'][] = "⚠️  Organization model may be missing some email fields";
    }
} else {
    $checks['failed'][] = "❌ Organization model NOT FOUND";
}

// Display results
echo "\n\n";
echo "Results:\n";
echo "========\n\n";

if (!empty($checks['passed'])) {
    echo "✅ PASSED (" . count($checks['passed']) . "):\n";
    foreach ($checks['passed'] as $pass) {
        echo "   {$pass}\n";
    }
    echo "\n";
}

if (!empty($checks['warnings'])) {
    echo "⚠️  WARNINGS (" . count($checks['warnings']) . "):\n";
    foreach ($checks['warnings'] as $warning) {
        echo "   {$warning}\n";
    }
    echo "\n";
}

if (!empty($checks['failed'])) {
    echo "❌ FAILED (" . count($checks['failed']) . "):\n";
    foreach ($checks['failed'] as $fail) {
        echo "   {$fail}\n";
    }
    echo "\n";
}

// Summary
echo "Summary:\n";
echo "--------\n";
echo "Passed: " . count($checks['passed']) . "\n";
echo "Warnings: " . count($checks['warnings']) . "\n";
echo "Failed: " . count($checks['failed']) . "\n\n";

// Final verdict
if (empty($checks['failed'])) {
    echo "✅ EMAIL SETUP COMPLETE!\n\n";
    echo "Next Steps:\n";
    echo "1. Configure .env email settings (if warnings above)\n";
    echo "2. Test by creating an organization (system admin)\n";
    echo "3. Test by creating an employee (org admin)\n";
    echo "4. Check email inbox for welcome emails\n";
    exit(0);
} else {
    echo "❌ EMAIL SETUP INCOMPLETE!\n";
    echo "Please fix the failed checks above.\n";
    exit(1);
}
