@extends('layouts.app')

@section('title', 'Performance Reviews')
@section('page-title', 'Performance Reviews')

@section('content')
<div class="container-fluid">
    <x-page-header title="Performance Reviews" description="Manage employee performance reviews">
        <x-slot:actions>
            @can('create_performance')
            <a href="{{ route('performance.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> New Review
            </a>
            @endcan
        </x-slot:actions>
    </x-page-header>

    <x-alert type="success" :message="session('success')" />

    <!-- Statistics Cards -->
    <x-stats-grid :stats="[
        ['value' => $stats['total_reviews'], 'label' => 'Total Reviews', 'icon' => 'star-fill', 'type' => 'primary'],
        ['value' => $stats['average_rating'] . '/5', 'label' => 'Average Rating', 'icon' => 'graph-up', 'type' => 'success'],
        ['value' => $stats['excellent_count'], 'label' => 'Excellent (≥4.5)', 'icon' => 'trophy-fill', 'type' => 'success'],
        ['value' => $stats['needs_improvement'], 'label' => 'Needs Improvement (<3)', 'icon' => 'exclamation-triangle-fill', 'type' => 'warning'],
        ['value' => $stats['pending_reviews'], 'label' => 'Pending', 'icon' => 'clock', 'type' => 'info'],
        ['value' => $stats['overdue_reviews'], 'label' => 'Overdue', 'icon' => 'exclamation-circle', 'type' => 'danger'],
    ]" columns="6" />

    <!-- Filters & Actions -->
    <div class="table-card mb-4">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h5 class="mb-0">Performance Reviews</h5>
            <div class="btn-group">
                @can('view_performance')
                <a href="{{ route('performance.goals.index') }}" class="btn btn-outline-primary">
                    <i class="bi bi-bullseye"></i> Goals
                </a>
                @endcan
                @can('view_performance')
                <a href="{{ route('performance.self-assessments.index') }}" class="btn btn-outline-success">
                    <i class="bi bi-person-check"></i> Self Assessments
                </a>
                @endcan
            </div>
        </div>

        <!-- Filter Form -->
        <form method="GET" class="row g-3 mb-4">
            <div class="col-md-2">
                <label class="form-label">Employee</label>
                <select name="employee" class="form-select">
                    <option value="">All Employees</option>
                    @foreach($employees as $employee)
                        <option value="{{ $employee->id }}" {{ request('employee') == $employee->id ? 'selected' : '' }}>
                            {{ $employee->full_name }}
                        </option>
                    @endforeach
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Period</label>
                <select name="period" class="form-select">
                    <option value="">All Periods</option>
                    <option value="Q1" {{ request('period') == 'Q1' ? 'selected' : '' }}>Q1</option>
                    <option value="Q2" {{ request('period') == 'Q2' ? 'selected' : '' }}>Q2</option>
                    <option value="Q3" {{ request('period') == 'Q3' ? 'selected' : '' }}>Q3</option>
                    <option value="Q4" {{ request('period') == 'Q4' ? 'selected' : '' }}>Q4</option>
                    <option value="Mid-Year" {{ request('period') == 'Mid-Year' ? 'selected' : '' }}>Mid-Year</option>
                    <option value="Yearly" {{ request('period') == 'Yearly' ? 'selected' : '' }}>Yearly</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Type</label>
                <select name="review_type" class="form-select">
                    <option value="">All Types</option>
                    <option value="annual" {{ request('review_type') == 'annual' ? 'selected' : '' }}>Annual</option>
                    <option value="mid_year" {{ request('review_type') == 'mid_year' ? 'selected' : '' }}>Mid-Year</option>
                    <option value="quarterly" {{ request('review_type') == 'quarterly' ? 'selected' : '' }}>Quarterly</option>
                    <option value="probation" {{ request('review_type') == 'probation' ? 'selected' : '' }}>Probation</option>
                    <option value="project_based" {{ request('review_type') == 'project_based' ? 'selected' : '' }}>Project Based</option>
                </select>
            </div>
            <div class="col-md-1">
                <label class="form-label">Rating</label>
                <select name="rating" class="form-select">
                    <option value="">All</option>
                    <option value="5" {{ request('rating') == '5' ? 'selected' : '' }}>5</option>
                    <option value="4" {{ request('rating') == '4' ? 'selected' : '' }}>4</option>
                    <option value="3" {{ request('rating') == '3' ? 'selected' : '' }}>3</option>
                    <option value="2" {{ request('rating') == '2' ? 'selected' : '' }}>2</option>
                    <option value="1" {{ request('rating') == '1' ? 'selected' : '' }}>1</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Status</label>
                <select name="status" class="form-select">
                    <option value="">All Status</option>
                    <option value="draft" {{ request('status') == 'draft' ? 'selected' : '' }}>Draft</option>
                    <option value="submitted" {{ request('status') == 'submitted' ? 'selected' : '' }}>Submitted</option>
                    <option value="acknowledged" {{ request('status') == 'acknowledged' ? 'selected' : '' }}>Acknowledged</option>
                    <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                </select>
            </div>
            <div class="col-md-1">
                <label class="form-label">Year</label>
                <select name="year" class="form-select">
                    <option value="">All</option>
                    @foreach($years as $year)
                        <option value="{{ $year }}" {{ request('year') == $year ? 'selected' : '' }}>{{ $year }}</option>
                    @endforeach
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">&nbsp;</label>
                <button type="submit" class="btn btn-primary w-100">
                    <i class="bi bi-funnel"></i> Filter
                </button>
            </div>
        </form>

        <!-- Reviews Table -->
        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                    <tr>
                        <th>Employee</th>
                        <th>Reviewer</th>
                        <th>Review Date</th>
                        <th>Period</th>
                        <th>Type</th>
                        <th>Overall Rating</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($reviews as $review)
                        <tr>
                            <td>
                                <div class="d-flex align-items-center">
                                    @if($review->employee->photo)
                                        <img src="{{ asset('storage/' . $review->employee->photo) }}"
                                             alt="{{ $review->employee->full_name }}"
                                             class="rounded-circle me-2"
                                             style="width: 32px; height: 32px; object-fit: cover;">
                                    @else
                                        <div class="rounded-circle me-2 bg-secondary text-white d-flex align-items-center justify-content-center"
                                             style="width: 32px; height: 32px; font-size: 14px;">
                                            {{ substr($review->employee->first_name, 0, 1) }}{{ substr($review->employee->last_name, 0, 1) }}
                                        </div>
                                    @endif
                                    <div>
                                        <div class="fw-bold">{{ $review->employee->full_name }}</div>
                                        <small class="text-muted">{{ $review->employee->employee_code }}</small>
                                    </div>
                                </div>
                            </td>
                            <td>{{ $review->reviewer->full_name ?? 'N/A' }}</td>
                            <td>{{ \Carbon\Carbon::parse($review->review_date)->format('M d, Y') }}</td>
                            <td><span class="badge bg-info">{{ $review->review_period }}</span></td>
                            <td><span class="badge bg-{{ $review->review_type_color }}">{{ ucfirst(str_replace('_', ' ', $review->review_type)) }}</span></td>
                            <td>
                                <div class="d-flex align-items-center">
                                    <div class="me-2">{{ number_format($review->overall_rating, 1) }}</div>
                                    <div>
                                        @for($i = 1; $i <= 5; $i++)
                                            @if($i <= $review->overall_rating)
                                                <i class="bi bi-star-fill text-warning"></i>
                                            @else
                                                <i class="bi bi-star text-muted"></i>
                                            @endif
                                        @endfor
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="badge bg-{{ $review->status === 'completed' ? 'success' : ($review->status === 'draft' ? 'secondary' : 'primary') }}">
                                    {{ ucfirst($review->status) }}
                                </span>
                                @if($review->isOverdue())
                                    <span class="badge bg-danger ms-1">Overdue</span>
                                @endif
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    @can('view', $review)
                                    <a href="{{ route('performance.show', $review) }}"
                                       class="btn btn-outline-primary"
                                       title="View">
                                        <i class="bi bi-eye"></i>
                                    </a>
                                    @endcan
                                    @can('update', $review)
                                    <a href="{{ route('performance.edit', $review) }}"
                                       class="btn btn-outline-secondary"
                                       title="Edit">
                                        <i class="bi bi-pencil"></i>
                                    </a>
                                    @endcan
                                    @can('delete', $review)
                                    <form action="{{ route('performance.destroy', $review) }}"
                                          method="POST"
                                          class="d-inline"
                                          onsubmit="return confirm('Are you sure you want to delete this review?');">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit"
                                                class="btn btn-outline-danger"
                                                title="Delete">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </form>
                                    @endcan
                                </div>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="8" class="text-center py-4 text-muted">
                                <i class="bi bi-inbox fs-1 d-block mb-2"></i>
                                No performance reviews found.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="mt-3">
            {{ $reviews->links() }}
        </div>
    </div>
</div>

@if(session('success'))
    <div class="toast-container position-fixed bottom-0 end-0 p-3">
        <div class="toast show" role="alert">
            <div class="toast-header bg-success text-white">
                <i class="bi bi-check-circle me-2"></i>
                <strong class="me-auto">Success</strong>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
            <div class="toast-body">
                {{ session('success') }}
            </div>
        </div>
    </div>
@endif
@endsection
