@extends('layouts.app')

@section('title', 'Payroll Management')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <x-page-header title="Payroll Management" description="Manage employee payroll records">
                <x-slot:actions>
                    <div class="btn-group">
                        <a href="{{ route('payroll.calculator') }}" class="btn btn-info">
                            <i class="bi bi-calculator"></i> Payroll Calculator
                        </a>
                        <a href="{{ route('payroll.create') }}" class="btn btn-primary">
                            <i class="bi bi-plus"></i> Generate Individual Payroll
                        </a>
                        <a href="{{ route('payroll.bulk') }}" class="btn btn-success">
                            <i class="bi bi-users"></i> Bulk Generate Payroll
                        </a>
                    </div>
                </x-slot:actions>
            </x-page-header>

            <x-alert type="success" :message="session('success')" />
            
            @if($errors->any())
                <x-alert type="danger">
                    @foreach($errors->all() as $error)
                        <div>{{ $error }}</div>
                    @endforeach
                </x-alert>
            @endif

            <!-- Filters -->
            <x-filter-form :action="route('payroll.index')">
                <div class="col-md-2">
                    <label for="year" class="form-label">Year</label>
                    <select name="year" id="year" class="form-select">
                        <option value="">All Years</option>
                        @foreach($years as $year)
                            <option value="{{ $year }}" {{ request('year') == $year ? 'selected' : '' }}>
                                {{ $year }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="month" class="form-label">Month</label>
                    <select name="month" id="month" class="form-select">
                        <option value="">All Months</option>
                        @foreach($months as $num => $name)
                            <option value="{{ $num }}" {{ request('month') == $num ? 'selected' : '' }}>
                                {{ $name }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="employment_type" class="form-label">Employment Type</label>
                    <select name="employment_type" id="employment_type" class="form-select">
                        <option value="">All Types</option>
                        @foreach($employmentTypes as $type)
                            <option value="{{ $type }}" {{ request('employment_type') == $type ? 'selected' : '' }}>
                                {{ ucfirst($type) }}
                            </option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label for="status" class="form-label">Status</label>
                    <select name="status" id="status" class="form-select">
                        <option value="">All Statuses</option>
                        @foreach($statuses as $status)
                            <option value="{{ $status }}" {{ request('status') == $status ? 'selected' : '' }}>
                                {{ ucfirst($status) }}
                            </option>
                        @endforeach
                    </select>
                </div>
            </x-filter-form>

            <!-- Payroll Table -->
            <div class="card">
                <div class="card-body">
                    @if($payrolls->count() > 0)
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>Payroll #</th>
                                        <th>Employee</th>
                                        <th>Employment Type</th>
                                        <th>Period</th>
                                        <th>Gross Salary</th>
                                        <th>Net Salary</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($payrolls as $payroll)
                                        <tr>
                                            <td>
                                                <code>{{ $payroll->payroll_number }}</code>
                                            </td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div>
                                                        <strong>{{ $payroll->employee->full_name }}</strong><br>
                                                        <small class="text-muted">{{ $payroll->employee->employee_code }}</small>
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge bg-info">{{ ucfirst($payroll->employment_type) }}</span>
                                            </td>
                                            <td>
                                                {{ date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)) }}
                                            </td>
                                            <td>
                                                <strong>KES {{ number_format($payroll->gross_salary, 2) }}</strong>
                                            </td>
                                            <td>
                                                <strong class="text-success">KES {{ number_format($payroll->net_salary, 2) }}</strong>
                                            </td>
                                            <td>
                                                @php
                                                    $statusColors = [
                                                        'draft' => 'secondary',
                                                        'approved' => 'warning',
                                                        'processed' => 'info',
                                                        'paid' => 'success'
                                                    ];
                                                @endphp
                                                <span class="badge bg-{{ $statusColors[$payroll->status] ?? 'secondary' }}">
                                                    {{ ucfirst($payroll->status) }}
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="{{ route('payroll.show', $payroll) }}" class="btn btn-outline-primary" title="View Details">
                                                        <i class="bi bi-eye"></i>
                                                    </a>
                                                    @if($payroll->status === 'processed')
                                                        <button type="button" class="btn btn-outline-success" data-bs-toggle="modal" data-bs-target="#markPaidModal{{ $payroll->id }}" title="Mark as Paid">
                                                            <i class="bi bi-check"></i>
                                                        </button>
                                                    @endif
                                                    @if($payroll->status === 'draft')
                                                        <form method="POST" action="{{ route('payroll.destroy', $payroll) }}" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this payroll?')">
                                                            @csrf
                                                            @method('DELETE')
                                                            <button type="submit" class="btn btn-outline-danger" title="Delete">
                                                                <i class="bi bi-trash"></i>
                                                            </button>
                                                        </form>
                                                    @endif
                                                </div>
                                            </td>
                                        </tr>

                                        <!-- Mark as Paid Modal -->
                                        @if($payroll->status === 'processed')
                                            <x-modal id="markPaidModal{{ $payroll->id }}" title="Mark Payroll as Paid">
                                                <form method="POST" action="{{ route('payroll.markPaid', $payroll) }}">
                                                    @csrf
                                                    <div class="mb-3">
                                                        <label for="payment_date{{ $payroll->id }}" class="form-label">Payment Date</label>
                                                        <input type="date" class="form-control" id="payment_date{{ $payroll->id }}" name="payment_date" value="{{ date('Y-m-d') }}" required>
                                                    </div>
                                                    <div class="mb-3">
                                                        <label for="payment_method{{ $payroll->id }}" class="form-label">Payment Method</label>
                                                        <select class="form-select" id="payment_method{{ $payroll->id }}" name="payment_method" required>
                                                            <option value="">Select Payment Method</option>
                                                            <option value="bank_transfer">Bank Transfer</option>
                                                            <option value="cash">Cash</option>
                                                            <option value="cheque">Cheque</option>
                                                            <option value="mobile_money">Mobile Money</option>
                                                        </select>
                                                    </div>
                                                    
                                                    <x-slot:footer>
                                                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                        <button type="submit" class="btn btn-success">Mark as Paid</button>
                                                    </x-slot:footer>
                                                </form>
                                            </x-modal>
                                        @endif
                                    @endforeach
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <div class="d-flex justify-content-center mt-4">
                            {{ $payrolls->appends(request()->query())->links() }}
                        </div>
                    @else
                        <div class="text-center py-5">
                            <i class="bi bi-money-bill-wave fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No payroll records found</h5>
                            <p class="text-muted">Start by generating payroll for your employees.</p>
                            <a href="{{ route('payroll.bulk') }}" class="btn btn-primary">
                                <i class="bi bi-plus"></i> Generate Payroll
                            </a>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>
@endsection