@extends('layouts.app')

@section('title', 'Request Loan')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-md-8 offset-md-2">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title mb-0">Request a Loan</h3>
                </div>
                <div class="card-body">
                    <!-- Eligibility Info -->
                    <div class="alert alert-info">
                        <h6><i class="bi bi-info-circle"></i> Loan Eligibility</h6>
                        <ul class="mb-0">
                            <li>Maximum loan amount: <strong>KES {{ number_format($loanSettings->max_loan_amount, 2) }}</strong></li>
                            <li>Maximum term: <strong>{{ $loanSettings->max_loan_term_months }} months</strong></li>
                            <li>Your maximum eligible amount: <strong>KES {{ number_format($employee->basic_salary * $loanSettings->loan_to_salary_ratio, 2) }}</strong> ({{ $loanSettings->loan_to_salary_ratio }}x your salary)</li>
                        </ul>
                    </div>

                    @if($errors->has('eligibility'))
                        <div class="alert alert-danger">
                            <h6><i class="bi bi-exclamation-triangle"></i> Eligibility Issues:</h6>
                            <ul class="mb-0">
                                @foreach($errors->get('eligibility') as $error)
                                    @foreach($error as $message)
                                        <li>{{ $message }}</li>
                                    @endforeach
                                @endforeach
                            </ul>
                        </div>
                    @endif

                    <form action="{{ route('loans.submit-request') }}" method="POST">
                        @csrf
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="loan_type" class="form-label">Loan Type <span class="text-danger">*</span></label>
                                <select name="loan_type" id="loan_type" class="form-select @error('loan_type') is-invalid @enderror" required>
                                    <option value="">Select Type</option>
                                    <option value="salary_advance" {{ old('loan_type') == 'salary_advance' ? 'selected' : '' }}>
                                        Salary Advance ({{ $loanSettings->getInterestRateForType('salary_advance') }}% interest)
                                    </option>
                                    <option value="emergency" {{ old('loan_type') == 'emergency' ? 'selected' : '' }}>
                                        Emergency ({{ $loanSettings->getInterestRateForType('emergency') }}% interest)
                                    </option>
                                    <option value="personal" {{ old('loan_type') == 'personal' ? 'selected' : '' }}>
                                        Personal ({{ $loanSettings->getInterestRateForType('personal') }}% interest)
                                    </option>
                                    <option value="education" {{ old('loan_type') == 'education' ? 'selected' : '' }}>
                                        Education ({{ $loanSettings->getInterestRateForType('education') }}% interest)
                                    </option>
                                    <option value="housing" {{ old('loan_type') == 'housing' ? 'selected' : '' }}>
                                        Housing ({{ $loanSettings->getInterestRateForType('housing') }}% interest)
                                    </option>
                                    <option value="other" {{ old('loan_type') == 'other' ? 'selected' : '' }}>
                                        Other ({{ $loanSettings->getInterestRateForType('other') }}% interest)
                                    </option>
                                </select>
                                @error('loan_type')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>

                            <div class="col-md-6 mb-3">
                                <label for="principal_amount" class="form-label">Loan Amount (KES) <span class="text-danger">*</span></label>
                                <input type="number" class="form-control @error('principal_amount') is-invalid @enderror" 
                                       id="principal_amount" name="principal_amount" 
                                       value="{{ old('principal_amount') }}" 
                                       step="0.01" min="1" max="{{ $loanSettings->max_loan_amount }}" required>
                                @error('principal_amount')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-muted">Maximum: KES {{ number_format($loanSettings->max_loan_amount, 2) }}</small>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="repayment_period_months" class="form-label">Repayment Period (Months) <span class="text-danger">*</span></label>
                            <input type="number" class="form-control @error('repayment_period_months') is-invalid @enderror" 
                                   id="repayment_period_months" name="repayment_period_months" 
                                   value="{{ old('repayment_period_months', 12) }}" 
                                   min="1" max="{{ $loanSettings->max_loan_term_months }}" required>
                            @error('repayment_period_months')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <small class="text-muted">Maximum: {{ $loanSettings->max_loan_term_months }} months</small>
                        </div>

                        <div class="mb-3">
                            <label for="purpose" class="form-label">Purpose/Reason <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('purpose') is-invalid @enderror" 
                                      id="purpose" name="purpose" rows="4" required 
                                      placeholder="Please explain why you need this loan...">{{ old('purpose') }}</textarea>
                            @error('purpose')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Loan Calculator Preview -->
                        <div class="card bg-light mb-3">
                            <div class="card-body">
                                <h6>Estimated Loan Details:</h6>
                                <div id="loan-preview">
                                    <p class="text-muted">Enter loan amount and period to see estimates</p>
                                </div>
                            </div>
                        </div>

                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" id="agree_terms" required>
                            <label class="form-check-label" for="agree_terms">
                                I agree to the loan terms and conditions, and authorize automatic deductions from my salary
                            </label>
                        </div>

                        <div class="d-flex justify-content-end">
                            <a href="{{ route('loans.my-loans') }}" class="btn btn-secondary me-2">Cancel</a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-send"></i> Submit Request
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const amountInput = document.getElementById('principal_amount');
    const periodInput = document.getElementById('repayment_period_months');
    const typeSelect = document.getElementById('loan_type');
    const previewDiv = document.getElementById('loan-preview');

    const interestRates = {
        'salary_advance': {{ $loanSettings->getInterestRateForType('salary_advance') }},
        'emergency': {{ $loanSettings->getInterestRateForType('emergency') }},
        'personal': {{ $loanSettings->getInterestRateForType('personal') }},
        'education': {{ $loanSettings->getInterestRateForType('education') }},
        'housing': {{ $loanSettings->getInterestRateForType('housing') }},
        'other': {{ $loanSettings->getInterestRateForType('other') }}
    };

    function calculateLoan() {
        const principal = parseFloat(amountInput.value) || 0;
        const months = parseInt(periodInput.value) || 0;
        const loanType = typeSelect.value;

        if (principal > 0 && months > 0 && loanType) {
            const annualRate = interestRates[loanType] / 100;
            const monthlyRate = annualRate / 12;
            
            let totalAmount, monthlyPayment;
            
            if (monthlyRate === 0) {
                totalAmount = principal;
                monthlyPayment = principal / months;
            } else {
                monthlyPayment = principal * (monthlyRate * Math.pow(1 + monthlyRate, months)) / 
                                (Math.pow(1 + monthlyRate, months) - 1);
                totalAmount = monthlyPayment * months;
            }

            const totalInterest = totalAmount - principal;

            previewDiv.innerHTML = `
                <table class="table table-sm mb-0">
                    <tr>
                        <td>Principal Amount:</td>
                        <td class="text-end"><strong>KES ${principal.toLocaleString('en-US', {minimumFractionDigits: 2})}</strong></td>
                    </tr>
                    <tr>
                        <td>Interest Rate:</td>
                        <td class="text-end"><strong>${interestRates[loanType]}% per year</strong></td>
                    </tr>
                    <tr>
                        <td>Total Interest:</td>
                        <td class="text-end"><strong>KES ${totalInterest.toLocaleString('en-US', {minimumFractionDigits: 2})}</strong></td>
                    </tr>
                    <tr>
                        <td>Total Amount to Repay:</td>
                        <td class="text-end"><strong>KES ${totalAmount.toLocaleString('en-US', {minimumFractionDigits: 2})}</strong></td>
                    </tr>
                    <tr class="table-primary">
                        <td><strong>Monthly Deduction:</strong></td>
                        <td class="text-end"><strong>KES ${monthlyPayment.toLocaleString('en-US', {minimumFractionDigits: 2})}</strong></td>
                    </tr>
                </table>
            `;
        }
    }

    amountInput.addEventListener('input', calculateLoan);
    periodInput.addEventListener('input', calculateLoan);
    typeSelect.addEventListener('change', calculateLoan);
});
</script>
@endpush
@endsection
