@extends('layouts.app')

@section('title', 'Edit Leave Request')
@section('page-title', 'Edit Leave Request')

@section('content')
    <x-breadcrumb :items="[
        ['label' => 'Dashboard', 'url' => route('dashboard')],
        ['label' => 'Leave Requests', 'url' => auth()->user()->can('viewAny', App\Models\LeaveRequest::class) ? route('leaves.index') : null],
        ['label' => 'Details', 'url' => auth()->user()->can('view', $leaveRequest) ? route('leaves.show', $leaveRequest) : null],
        ['label' => 'Edit']
    ]" />

    <div class="row">
        <div class="col-md-8">
            <div class="table-card">
                <div class="mb-4">
                    <h4>Edit Leave Request</h4>
                    <p class="text-muted">Update leave request for {{ $leaveRequest->employee->full_name }}</p>
                    @if($leaveRequest->status != 'pending')
                        <x-alert type="warning">
                            This leave request has been {{ $leaveRequest->status }}. Changes may require re-approval.
                        </x-alert>
                    @endif
                </div>

                <form action="{{ route('leaves.update', $leaveRequest) }}" method="POST">
                    @csrf
                    @method('PUT')

                    <div class="row">
                        <div class="col-md-6">
                            <x-form.select 
                                name="employee_id" 
                                label="Employee" 
                                required
                                :value="$leaveRequest->employee_id"
                                :options="$employees->mapWithKeys(fn($e) => [$e->id => $e->full_name . ' (' . $e->employee_code . ')'])->toArray()" />
                        </div>

                        <div class="col-md-6">
                            <div class="mb-3">
                                <label for="leave_type_id" class="form-label">Leave Type <span class="text-danger">*</span></label>
                                <select class="form-select @error('leave_type_id') is-invalid @enderror"
                                        id="leave_type_id" name="leave_type_id" required>
                                    <option value="">Select Leave Type</option>
                                    @foreach($leaveTypes as $type)
                                        <option value="{{ $type->id }}"
                                                data-days="{{ $type->default_days }}"
                                                {{ old('leave_type_id', $leaveRequest->leave_type_id) == $type->id ? 'selected' : '' }}>
                                            {{ $type->name }} ({{ $type->default_days }} days)
                                        </option>
                                    @endforeach
                                </select>
                                @error('leave_type_id')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <div class="col-md-6">
                            <x-form.input 
                                type="date" 
                                name="start_date" 
                                label="Start Date" 
                                required
                                :value="$leaveRequest->start_date->format('Y-m-d')" />
                        </div>

                        <div class="col-md-6">
                            <x-form.input 
                                type="date" 
                                name="end_date" 
                                label="End Date" 
                                required
                                :value="$leaveRequest->end_date->format('Y-m-d')" />
                        </div>

                        <div class="col-md-12 mb-3">
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                <strong>Duration:</strong> <span id="duration-display">{{ $leaveRequest->number_of_days }} days</span>
                            </div>
                        </div>

                        <div class="col-md-12">
                            <x-form.textarea 
                                name="reason" 
                                label="Reason" 
                                required
                                rows="4"
                                :value="$leaveRequest->reason" />
                        </div>

                        <div class="col-md-12">
                            <x-form.textarea 
                                name="notes" 
                                label="Additional Notes"
                                rows="3"
                                :value="$leaveRequest->notes" />
                        </div>
                    </div>

                    <x-form.button-group 
                        :cancelUrl="auth()->user()->can('view', $leaveRequest) ? route('leaves.show', $leaveRequest) : route('leaves.index')"
                        submitText="Update Request"
                        submitIcon="bi-check-circle" />
                </form>
            </div>
        </div>

        <div class="col-md-4">
            <div class="table-card">
                <h6 class="mb-3"><i class="bi bi-info-circle me-2"></i>Leave Balance</h6>
                <div id="leave-balance-info">
                    @if($leaveRequest->employee->leaveBalances->count() > 0)
                        <table class="table table-sm">
                            @foreach($leaveRequest->employee->leaveBalances as $balance)
                                <tr>
                                    <td>
                                        <span class="badge" style="background-color: {{ $balance->leaveType->color }}">
                                            {{ $balance->leaveType->name }}
                                        </span>
                                    </td>
                                    <td class="text-end">
                                        <strong>{{ $balance->available_days }}</strong> / {{ $balance->total_days }}
                                    </td>
                                </tr>
                            @endforeach
                        </table>
                    @else
                        <p class="text-muted">No leave balance information</p>
                    @endif
                </div>
            </div>

            <div class="table-card mt-3">
                <h6 class="mb-3"><i class="bi bi-list-ul me-2"></i>Leave Types</h6>
                <div class="list-group list-group-flush">
                    @foreach($leaveTypes as $type)
                        <div class="list-group-item d-flex justify-content-between align-items-center">
                            <div>
                                <span class="badge" style="background-color: {{ $type->color }}"></span>
                                {{ $type->name }}
                            </div>
                            <span class="badge bg-secondary">{{ $type->default_days }} days</span>
                        </div>
                    @endforeach
                </div>
            </div>

            <div class="table-card mt-3">
                <h6 class="mb-3"><i class="bi bi-clock-history me-2"></i>Recent Leaves</h6>
                <div class="list-group list-group-flush">
                    @foreach($leaveRequest->employee->leaveRequests()->where('id', '!=', $leaveRequest->id)->latest()->take(5)->get() as $recent)
                        <div class="list-group-item px-0">
                            <div class="d-flex justify-content-between">
                                <small>{{ $recent->leaveType->name }}</small>
                                <x-status-badge :status="$recent->status" />
                            </div>
                            <small class="text-muted">
                                {{ $recent->start_date->format('M d') }} - {{ $recent->end_date->format('M d, Y') }}
                            </small>
                        </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
<script>
    // Calculate duration when dates change
    document.getElementById('start_date').addEventListener('change', calculateDuration);
    document.getElementById('end_date').addEventListener('change', calculateDuration);

    function calculateDuration() {
        const startDate = new Date(document.getElementById('start_date').value);
        const endDate = new Date(document.getElementById('end_date').value);

        if (startDate && endDate && endDate >= startDate) {
            const diffTime = Math.abs(endDate - startDate);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)) + 1;
            document.getElementById('duration-display').textContent = diffDays + ' days';
        } else {
            document.getElementById('duration-display').textContent = '0 days';
        }
    }

    // Load leave balance when employee changes
    document.getElementById('employee_id').addEventListener('change', function() {
        const employeeId = this.value;
        const leaveBalanceInfo = document.getElementById('leave-balance-info');

        if (!employeeId) {
            leaveBalanceInfo.innerHTML = '<p class="text-muted">Select an employee to view their leave balance</p>';
            return;
        }

        leaveBalanceInfo.innerHTML = '<div class="spinner-border spinner-border-sm"></div> Loading...';

        fetch(`/leaves/balance/${employeeId}`)
            .then(response => response.json())
            .then(data => {
                let html = '<table class="table table-sm">';
                data.forEach(balance => {
                    html += `
                        <tr>
                            <td>
                                <span class="badge" style="background-color: ${balance.color}">${balance.leave_type}</span>
                            </td>
                            <td class="text-end">
                                <strong>${balance.available}</strong> / ${balance.total}
                            </td>
                        </tr>
                    `;
                });
                html += '</table>';
                leaveBalanceInfo.innerHTML = html;
            })
            .catch(error => {
                leaveBalanceInfo.innerHTML = '<p class="text-danger">Error loading balance</p>';
            });
    });
</script>
@endpush
