@extends('layouts.app')

@section('title', 'New Leave Request')
@section('page-title', 'New Leave Request')

@section('content')
    <x-breadcrumb :items="[
        ['label' => 'Dashboard', 'url' => route('dashboard')],
        ['label' => 'Leave Requests', 'url' => auth()->user()->can('viewAny', App\Models\LeaveRequest::class) ? route('leaves.index') : null],
        ['label' => 'New Request']
    ]" />

    <div class="row">
        <div class="col-md-8">
            <div class="table-card">
                <form action="{{ route('leaves.store') }}" method="POST">
                    @csrf

                    <div class="row">
                        <div class="col-md-6">
                            @if($employees->count() == 1)
                                <x-form.input 
                                    name="employee_display" 
                                    label="Employee" 
                                    :value="$employees->first()->full_name . ' (' . $employees->first()->employee_code . ')'" 
                                    readonly
                                    help="You are creating a leave request for yourself" />
                                <input type="hidden" name="employee_id" value="{{ $employees->first()->id }}">
                            @else
                                <x-form.select 
                                    name="employee_id" 
                                    label="Employee" 
                                    required
                                    placeholder="Select Employee"
                                    :value="old('employee_id', $currentEmployee?->id)"
                                    :options="$employees->mapWithKeys(fn($e) => [$e->id => $e->full_name . ' (' . $e->employee_code . ')'])->toArray()" />
                            @endif
                        </div>

                        <div class="col-md-6">
                            <x-form.select 
                                name="leave_type_id" 
                                label="Leave Type" 
                                required
                                placeholder="Select Leave Type"
                                :options="$leaveTypes->mapWithKeys(fn($t) => [$t->id => $t->name . ' (' . $t->days_allowed . ' days)'])->toArray()" />
                        </div>

                        <div class="col-md-6">
                            <x-form.date name="start_date" label="Start Date" required />
                        </div>

                        <div class="col-md-6 mb-3">
                            <label for="end_date" class="form-label">End Date <span class="text-danger">*</span></label>
                            <input type="date" class="form-control @error('end_date') is-invalid @enderror"
                                   id="end_date" name="end_date" value="{{ old('end_date') }}" required>
                            @error('end_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="col-md-12 mb-3">
                            <div class="alert alert-info">
                                <i class="bi bi-info-circle me-2"></i>
                                <strong>Duration:</strong> <span id="duration-display">0 days</span>
                            </div>
                        </div>

                        <div class="col-md-12 mb-3">
                            <label for="reason" class="form-label">Reason <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('reason') is-invalid @enderror"
                                      id="reason" name="reason" rows="4" required>{{ old('reason') }}</textarea>
                            @error('reason')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>

                    <div class="mt-4">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-check-circle"></i> Submit Request
                        </button>
                        @can('viewAny', App\Models\LeaveRequest::class)
                        <a href="{{ route('leaves.index') }}" class="btn btn-secondary">
                            <i class="bi bi-x-circle"></i> Cancel
                        </a>
                        @else
                        <a href="{{ route('dashboard') }}" class="btn btn-secondary">
                            <i class="bi bi-x-circle"></i> Cancel
                        </a>
                        @endcan
                    </div>
                </form>
            </div>
        </div>

        <div class="col-md-4">
            <div class="table-card">
                <h6 class="mb-3"><i class="bi bi-info-circle me-2"></i>Leave Balance</h6>
                <div id="leave-balance-info">
                    @if($employees->count() == 1)
                        <p class="text-muted">Loading your leave balance...</p>
                    @else
                        <p class="text-muted">Select an employee to view their leave balance</p>
                    @endif
                </div>
            </div>

            <div class="table-card mt-3">
                <h6 class="mb-3"><i class="bi bi-list-ul me-2"></i>Leave Types</h6>
                <div class="list-group list-group-flush">
                    @foreach($leaveTypes as $type)
                        <div class="list-group-item d-flex justify-content-between align-items-center">
                            <div>
                                <span class="badge" style="background-color: {{ $type->color }}"></span>
                                {{ $type->name }}
                            </div>
                            <span class="badge bg-secondary">{{ $type->days_allowed }} days</span>
                        </div>
                    @endforeach
                </div>
            </div>
        </div>
    </div>
@endsection

@push('scripts')
<script>
    // Calculate duration when dates change
    document.getElementById('start_date').addEventListener('change', calculateDuration);
    document.getElementById('end_date').addEventListener('change', calculateDuration);

    function calculateDuration() {
        const startDate = new Date(document.getElementById('start_date').value);
        const endDate = new Date(document.getElementById('end_date').value);

        if (startDate && endDate && endDate >= startDate) {
            const diffTime = Math.abs(endDate - startDate);
            const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)) + 1;
            document.getElementById('duration-display').textContent = diffDays + ' days';
        } else {
            document.getElementById('duration-display').textContent = '0 days';
        }
    }

    // Load leave balance when employee changes
    const employeeSelect = document.getElementById('employee_id');
    if (employeeSelect) {
        employeeSelect.addEventListener('change', loadLeaveBalance);
        
        // Load balance immediately if employee is already selected
        if (employeeSelect.value) {
            loadLeaveBalance();
        }
    } else {
        // For single employee case, get the hidden input value
        const hiddenEmployeeId = document.querySelector('input[name="employee_id"]');
        if (hiddenEmployeeId && hiddenEmployeeId.value) {
            loadLeaveBalanceForEmployee(hiddenEmployeeId.value);
        }
    }

    function loadLeaveBalance() {
        const employeeId = document.getElementById('employee_id').value;
        loadLeaveBalanceForEmployee(employeeId);
    }

    function loadLeaveBalanceForEmployee(employeeId) {
        const leaveBalanceInfo = document.getElementById('leave-balance-info');

        if (!employeeId) {
            leaveBalanceInfo.innerHTML = '<p class="text-muted">Select an employee to view their leave balance</p>';
            return;
        }

        leaveBalanceInfo.innerHTML = '<div class="spinner-border spinner-border-sm"></div> Loading...';

        fetch(`/leaves/balance/${employeeId}`)
            .then(response => response.json())
            .then(data => {
                let html = '<table class="table table-sm">';
                data.forEach(balance => {
                    html += `
                        <tr>
                            <td>${balance.leave_type}</td>
                            <td class="text-end">
                                <span class="badge bg-success">${balance.available}</span> /
                                ${balance.total}
                            </td>
                        </tr>
                    `;
                });
                html += '</table>';
                leaveBalanceInfo.innerHTML = html;
            })
            .catch(error => {
                leaveBalanceInfo.innerHTML = '<p class="text-danger">Error loading balance</p>';
            });
    }
</script>
@endpush
