@extends('layouts.app')

@section('title', 'Attendance')
@section('page-title', 'Attendance Management')

@section('content')
    <x-page-header 
        title="Attendance Records" 
        description="Track and manage employee attendance">
        <x-slot:actions>
            @can('create', App\Models\Attendance::class)
            <a href="{{ route('attendance.create') }}" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Mark Attendance
            </a>
            @endcan
        </x-slot:actions>
    </x-page-header>

    <x-alert type="success" :message="session('success')" />

    <x-stats-grid :stats="[
        ['value' => $stats['present'], 'label' => 'Present Today', 'icon' => 'check-circle', 'type' => 'success'],
        ['value' => $stats['absent'], 'label' => 'Absent Today', 'icon' => 'x-circle', 'type' => 'danger'],
        ['value' => $stats['late'], 'label' => 'Late Today', 'icon' => 'clock', 'type' => 'warning'],
        ['value' => $stats['on_leave'], 'label' => 'On Leave', 'icon' => 'calendar-check', 'type' => 'info'],
    ]" />

    <x-filter-form :action="route('attendance.index')">
        <div class="col-md-3">
            <label class="form-label">From Date</label>
            <input type="date" class="form-control" name="from_date"
                   value="{{ request('from_date', now()->startOfMonth()->format('Y-m-d')) }}">
        </div>
        <div class="col-md-3">
            <label class="form-label">To Date</label>
            <input type="date" class="form-control" name="to_date"
                   value="{{ request('to_date', now()->format('Y-m-d')) }}">
        </div>
        <div class="col-md-2">
            <label class="form-label">Status</label>
            <select class="form-select" name="status">
                <option value="">All Status</option>
                <option value="present" {{ request('status') == 'present' ? 'selected' : '' }}>Present</option>
                <option value="absent" {{ request('status') == 'absent' ? 'selected' : '' }}>Absent</option>
                <option value="late" {{ request('status') == 'late' ? 'selected' : '' }}>Late</option>
                <option value="on_leave" {{ request('status') == 'on_leave' ? 'selected' : '' }}>On Leave</option>
            </select>
        </div>
        <div class="col-md-2">
            <label class="form-label">Department</label>
            <select class="form-select" name="department">
                <option value="">All Departments</option>
                @foreach($departments as $department)
                    <option value="{{ $department->id }}" {{ request('department') == $department->id ? 'selected' : '' }}>
                        {{ $department->name }}
                    </option>
                @endforeach
            </select>
        </div>
    </x-filter-form>

    <div class="row">
        <div class="col-md-12">
            <div class="table-card">

                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Employee</th>
                                <th>Department</th>
                                <th>Date</th>
                                <th>Check In</th>
                                <th>Check Out</th>
                                <th>Work Hours</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($attendances as $attendance)
                                <tr>
                                    <td>
                                        <x-user-avatar :user="$attendance->employee" show-name show-details />
                                    </td>
                                    <td>{{ $attendance->employee->department->name ?? 'N/A' }}</td>
                                    <td>{{ $attendance->date->format('M d, Y') }}</td>
                                    <td>
                                        <span class="{{ $attendance->is_late ? 'text-danger' : 'text-success' }}">
                                            {{ $attendance->check_in ?? 'N/A' }}
                                            @if($attendance->is_late)
                                                <i class="bi bi-exclamation-circle" title="Late arrival"></i>
                                            @endif
                                        </span>
                                    </td>
                                    <td>{{ $attendance->check_out ?? 'Not yet' }}</td>
                                    <td>
                                        @if($attendance->work_hours)
                                            {{ number_format($attendance->work_hours, 2) }} hrs
                                        @else
                                            N/A
                                        @endif
                                    </td>
                                    <td>
                                        <x-status-badge :status="$attendance->status" />
                                    </td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            @can('view', $attendance)
                                            <a href="{{ route('attendance.show', $attendance) }}" class="btn btn-outline-primary" title="View">
                                                <i class="bi bi-eye"></i>
                                            </a>
                                            @endcan
                                            @can('update', $attendance)
                                            <a href="{{ route('attendance.edit', $attendance) }}" class="btn btn-outline-secondary" title="Edit">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            @endcan
                                            @if(!$attendance->check_out && $attendance->status == 'present')
                                                @can('update', $attendance)
                                                <form action="{{ route('attendance.checkout', $attendance) }}" method="POST" class="d-inline">
                                                    @csrf
                                                    <button type="submit" class="btn btn-outline-warning" title="Check Out">
                                                        <i class="bi bi-box-arrow-right"></i>
                                                    </button>
                                                </form>
                                                @endcan
                                            @endif
                                            @can('delete', $attendance)
                                            <form action="{{ route('attendance.destroy', $attendance) }}" method="POST" class="d-inline">
                                                @csrf
                                                @method('DELETE')
                                                <button type="submit" class="btn btn-outline-danger"
                                                        onclick="return confirm('Are you sure?')" title="Delete">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                            @endcan
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8">
                                        <x-empty-state 
                                            icon="calendar-x" 
                                            title="No attendance records found"
                                            description="No attendance records match your current filters" />
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                @if($attendances->hasPages())
                    <div class="d-flex justify-content-center mt-4">
                        {{ $attendances->links() }}
                    </div>
                @endif
            </div>
        </div>
    </div>
@endsection
