# Development Setup Guide

## Prerequisites

Before setting up the Qaya HRM system, ensure you have the following installed:

### Required Software

- **PHP** >= 8.2
- **Composer** (Latest version)
- **MySQL** or **MariaDB** >= 8.0
- **Node.js** >= 18.x
- **NPM** or **Yarn**
- **Git**

### Optional Tools

- **Redis** (for caching and queues)
- **Mailpit** or **Mailtrap** (for email testing)
- **Laravel Valet** or **Herd** (for macOS)
- **Docker** (for containerized development)

---

## Installation Steps

### 1. Clone the Repository

```bash
git clone https://github.com/your-org/qaya-hrm.git
cd qaya-hrm
```

### 2. Install PHP Dependencies

```bash
composer install
```

**If you encounter memory issues**:
```bash
COMPOSER_MEMORY_LIMIT=-1 composer install
```

### 3. Install Node Dependencies

```bash
npm install
```

**Or using Yarn**:
```bash
yarn install
```

### 4. Environment Configuration

**Copy the example environment file**:
```bash
cp .env.example .env
```

**Generate application key**:
```bash
php artisan key:generate
```

**Configure your .env file**:
```env
APP_NAME="Qaya HRM"
APP_ENV=local
APP_KEY=base64:... # Generated by key:generate
APP_DEBUG=true
APP_URL=http://localhost:8000

# Database Configuration
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=qaya_hrm
DB_USERNAME=root
DB_PASSWORD=your_password

# Mail Configuration (for development)
MAIL_MAILER=smtp
MAIL_HOST=mailpit
MAIL_PORT=1025
MAIL_USERNAME=null
MAIL_PASSWORD=null
MAIL_ENCRYPTION=null
MAIL_FROM_ADDRESS="noreply@qaya-hrm.local"
MAIL_FROM_NAME="${APP_NAME}"

# Queue Configuration
QUEUE_CONNECTION=database

# Session Configuration
SESSION_DRIVER=database
SESSION_LIFETIME=120

# Cache Configuration
CACHE_DRIVER=file
```

### 5. Create Database

**Using MySQL CLI**:
```bash
mysql -u root -p
```

```sql
CREATE DATABASE qaya_hrm CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
EXIT;
```

**Or using phpMyAdmin**:
1. Open phpMyAdmin
2. Click "New" to create database
3. Name: `qaya_hrm`
4. Collation: `utf8mb4_unicode_ci`
5. Click "Create"

### 6. Run Migrations

```bash
php artisan migrate
```

**Expected output**:
```
Migration table created successfully.
Migrating: 2014_10_12_000000_create_users_table
Migrated:  2014_10_12_000000_create_users_table (45.23ms)
Migrating: 2025_01_01_000001_create_hr_management_system_tables
Migrated:  2025_01_01_000001_create_hr_management_system_tables (123.45ms)
...
```

### 7. Seed Database

```bash
php artisan db:seed
```

**This will create**:
- Default SystemAdmin account
- Sample organization
- Default roles and permissions
- Sample employees (optional)
- Leave types
- Attendance settings

### 8. Create Storage Link

```bash
php artisan storage:link
```

### 9. Build Frontend Assets

**For development**:
```bash
npm run dev
```

**For production**:
```bash
npm run build
```

### 10. Start Development Server

```bash
php artisan serve
```

**Application will be available at**: `http://localhost:8000`

---

## Default Login Credentials

### SystemAdmin Account

- **URL**: `http://localhost:8000/systemadmin/login`
- **Email**: `systemadmin@example.com`
- **Password**: `password`

### Organization Admin Account

- **URL**: `http://localhost:8000/login`
- **Email**: `qayacreatives@gmail.com`
- **Password**: `password`

### Sample Employee Account

- **URL**: `http://localhost:8000/login`
- **Email**: `employee@example.com`
- **Password**: `password`

**⚠️ Important**: Change these passwords immediately in production!

---

## Development Workflow

### Running the Application

**Option 1: Artisan Serve (Simple)**
```bash
php artisan serve
```

**Option 2: Laravel Valet (macOS)**
```bash
cd qaya-hrm
valet link
valet secure qaya-hrm # Optional: Enable HTTPS
```
Access at: `https://qaya-hrm.test`

**Option 3: Laravel Herd (macOS/Windows)**
- Add project to Herd
- Access at: `http://qaya-hrm.test`

**Option 4: Docker (All platforms)**
```bash
./vendor/bin/sail up
```
Access at: `http://localhost`

### Frontend Development

**Watch for changes (Hot reload)**:
```bash
npm run dev
```

**Build for production**:
```bash
npm run build
```

### Queue Worker

**Start queue worker**:
```bash
php artisan queue:work
```

**Or use queue listener**:
```bash
php artisan queue:listen
```

### Scheduler (Cron Jobs)

**For development, run manually**:
```bash
php artisan schedule:work
```

**For production, add to crontab**:
```bash
* * * * * cd /path-to-your-project && php artisan schedule:run >> /dev/null 2>&1
```

---

## IDE Setup

### Visual Studio Code

**Recommended Extensions**:
- PHP Intelephense
- Laravel Extension Pack
- Laravel Blade Snippets
- Laravel goto view
- Tailwind CSS IntelliSense
- ESLint
- Prettier

**Settings (.vscode/settings.json)**:
```json
{
  "php.suggest.basic": false,
  "intelephense.files.associations": [
    "*.php",
    "*.blade.php"
  ],
  "emmet.includeLanguages": {
    "blade": "html"
  },
  "files.associations": {
    "*.blade.php": "blade"
  }
}
```

### PHPStorm

**Configuration**:
1. Enable Laravel Plugin
2. Set PHP interpreter to 8.2+
3. Configure Composer
4. Enable Blade syntax highlighting
5. Configure database connection

---

## Database Management

### Using Artisan Tinker

**Access database via Tinker**:
```bash
php artisan tinker
```

**Example commands**:
```php
// Get all users
User::all();

// Create a user
User::create([
    'name' => 'Test User',
    'email' => 'test@example.com',
    'password' => bcrypt('password'),
    'organization_id' => 1
]);

// Get organization
Organization::find(1);
```

### Database Seeding

**Seed specific seeder**:
```bash
php artisan db:seed --class=RoleSeeder
```

**Refresh database and seed**:
```bash
php artisan migrate:fresh --seed
```

**⚠️ Warning**: This will delete all data!

---

## Testing

### Running Tests

**Run all tests**:
```bash
php artisan test
```

**Run specific test**:
```bash
php artisan test --filter=EmployeeTest
```

**Run with coverage**:
```bash
php artisan test --coverage
```

### Creating Tests

**Create feature test**:
```bash
php artisan make:test EmployeeTest
```

**Create unit test**:
```bash
php artisan make:test EmployeeTest --unit
```

---

## Debugging

### Laravel Debugbar

**Install (if not included)**:
```bash
composer require barryvdh/laravel-debugbar --dev
```

**Enable in .env**:
```env
DEBUGBAR_ENABLED=true
```

### Laravel Telescope

**Install (optional)**:
```bash
composer require laravel/telescope --dev
php artisan telescope:install
php artisan migrate
```

**Access**: `http://localhost:8000/telescope`

### Logging

**View logs**:
```bash
tail -f storage/logs/laravel.log
```

**Clear logs**:
```bash
> storage/logs/laravel.log
```

---

## Common Commands

### Artisan Commands

```bash
# Clear caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear

# Optimize for production
php artisan optimize
php artisan config:cache
php artisan route:cache
php artisan view:cache

# Database
php artisan migrate
php artisan migrate:rollback
php artisan migrate:fresh
php artisan db:seed

# Create resources
php artisan make:model Employee
php artisan make:controller EmployeeController
php artisan make:migration create_employees_table
php artisan make:seeder EmployeeSeeder
php artisan make:request StoreEmployeeRequest
php artisan make:policy EmployeePolicy

# Custom commands
php artisan org:create "Company Name" admin@company.com "Admin Name"
php artisan org:list
php artisan test:multitenancy
```

### Composer Commands

```bash
# Install dependencies
composer install

# Update dependencies
composer update

# Dump autoload
composer dump-autoload

# Require package
composer require package/name

# Remove package
composer remove package/name
```

### NPM Commands

```bash
# Install dependencies
npm install

# Development build
npm run dev

# Production build
npm run build

# Watch for changes
npm run watch

# Update packages
npm update
```

---

## Troubleshooting

### Common Issues

**Issue**: "Class not found" error
```bash
composer dump-autoload
```

**Issue**: "No application encryption key"
```bash
php artisan key:generate
```

**Issue**: "Access denied for user"
- Check database credentials in .env
- Verify MySQL is running
- Test connection: `mysql -u root -p`

**Issue**: "Storage not writable"
```bash
chmod -R 775 storage bootstrap/cache
```

**Issue**: "Mix manifest not found"
```bash
npm run build
```

**Issue**: "SQLSTATE[HY000] [2002] Connection refused"
- Check MySQL is running
- Verify DB_HOST in .env
- Try `127.0.0.1` instead of `localhost`

**Issue**: "419 Page Expired" (CSRF token mismatch)
```bash
php artisan cache:clear
php artisan config:clear
```

---

## Performance Optimization

### Development

**Enable query logging**:
```php
// In AppServiceProvider
DB::listen(function($query) {
    Log::info($query->sql, $query->bindings);
});
```

**Use Laravel Debugbar**:
- Monitor queries
- Check execution time
- View memory usage

### Production

**Optimize autoloader**:
```bash
composer install --optimize-autoloader --no-dev
```

**Cache configuration**:
```bash
php artisan config:cache
php artisan route:cache
php artisan view:cache
```

**Enable OPcache** (php.ini):
```ini
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=20000
```

---

## Git Workflow

### Branch Strategy

```
main (production)
  ├── develop (development)
  │   ├── feature/employee-management
  │   ├── feature/payroll-system
  │   └── bugfix/attendance-calculation
  └── hotfix/security-patch
```

### Commit Messages

**Format**:
```
<type>(<scope>): <subject>

<body>

<footer>
```

**Types**:
- `feat`: New feature
- `fix`: Bug fix
- `docs`: Documentation
- `style`: Formatting
- `refactor`: Code restructuring
- `test`: Adding tests
- `chore`: Maintenance

**Example**:
```
feat(payroll): add Kenya tax calculation

- Implement PAYE calculation
- Add NHIF and NSSF deductions
- Include housing levy

Closes #123
```

---

## Environment-Specific Configuration

### Local Development

```env
APP_ENV=local
APP_DEBUG=true
DEBUGBAR_ENABLED=true
```

### Staging

```env
APP_ENV=staging
APP_DEBUG=false
DEBUGBAR_ENABLED=false
```

### Production

```env
APP_ENV=production
APP_DEBUG=false
DEBUGBAR_ENABLED=false
```

---

## Next Steps

After setup:

1. ✅ Change default passwords
2. ✅ Configure organization settings
3. ✅ Set up email configuration
4. ✅ Create departments and positions
5. ✅ Add employees
6. ✅ Configure attendance settings
7. ✅ Set up leave types
8. ✅ Configure payroll settings
9. ✅ Test all modules
10. ✅ Review security settings

---

## Additional Resources

- [Laravel Documentation](https://laravel.com/docs)
- [Bootstrap Documentation](https://getbootstrap.com/docs)
- [PHP Documentation](https://www.php.net/docs.php)
- [MySQL Documentation](https://dev.mysql.com/doc/)

---

**Last Updated**: February 2026
