<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Organization;
use App\Models\ThemeSetting;

class ThemeSettingsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * This seeder creates default theme settings for existing organizations
     * that don't have theme settings yet.
     */
    public function run(): void
    {
        $this->command->info('Creating default theme settings for organizations...');
        
        // Get all organizations without theme settings
        $organizations = Organization::whereDoesntHave('themeSetting')->get();
        
        if ($organizations->isEmpty()) {
            $this->command->info('All organizations already have theme settings.');
            return;
        }
        
        foreach ($organizations as $organization) {
            ThemeSetting::create([
                'organization_id' => $organization->id,
                'base_theme' => 'default',
                'is_active' => true,
            ]);
            
            $this->command->info("✓ Created default theme for: {$organization->name}");
        }
        
        $this->command->info("✓ Created theme settings for {$organizations->count()} organizations");
        
        // Optional: Create a sample customized theme for demonstration
        $this->createSampleCustomTheme();
    }
    
    /**
     * Create a sample customized theme for the first organization (optional)
     */
    protected function createSampleCustomTheme(): void
    {
        // Check if we want to create a demo theme
        if (!$this->command->confirm('Create a sample customized theme for demo purposes?', false)) {
            return;
        }
        
        $firstOrg = Organization::first();
        
        if (!$firstOrg) {
            $this->command->warn('No organizations found to create sample theme.');
            return;
        }
        
        // Update or create a customized theme
        ThemeSetting::updateOrCreate(
            ['organization_id' => $firstOrg->id],
            [
                'base_theme' => 'blue',
                'primary_color' => '#0066cc',
                'secondary_color' => '#0099ff',
                'accent_color' => '#00ccff',
                'sidebar_bg_color' => '#003366',
                'sidebar_hover_color' => '#0066cc',
                'primary_font' => 'Inter, system-ui, sans-serif',
                'heading_font' => 'Poppins, sans-serif',
                'is_active' => true,
            ]
        );
        
        $this->command->info("✓ Created sample customized theme for: {$firstOrg->name}");
    }
}
