<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\PerformanceCompetency;
use App\Models\PerformanceGoal;
use App\Models\PerformanceSelfAssessment;
use App\Models\Employee;
use App\Models\User;

class PerformanceEnhancementSeeder extends Seeder
{
    public function run(): void
    {
        // Create default competencies
        $competencies = [
            [
                'name' => 'Technical Skills',
                'description' => 'Proficiency in job-related technical skills and knowledge',
                'category' => 'technical',
                'skill_levels' => [
                    1 => 'Basic understanding, requires significant guidance',
                    2 => 'Developing skills, requires some guidance',
                    3 => 'Competent, works independently',
                    4 => 'Advanced skills, mentors others',
                    5 => 'Expert level, leads technical initiatives'
                ],
                'organization_id' => 1,
            ],
            [
                'name' => 'Communication',
                'description' => 'Ability to communicate effectively with team members and stakeholders',
                'category' => 'behavioral',
                'skill_levels' => [
                    1 => 'Struggles with clear communication',
                    2 => 'Basic communication skills',
                    3 => 'Communicates clearly and effectively',
                    4 => 'Excellent communicator, facilitates discussions',
                    5 => 'Outstanding communication, influences and inspires'
                ],
                'organization_id' => 1,
            ],
            [
                'name' => 'Leadership',
                'description' => 'Ability to lead, motivate, and guide team members',
                'category' => 'leadership',
                'skill_levels' => [
                    1 => 'No leadership experience or skills',
                    2 => 'Basic leadership potential',
                    3 => 'Demonstrates leadership in small projects',
                    4 => 'Strong leader, manages teams effectively',
                    5 => 'Exceptional leader, drives organizational change'
                ],
                'organization_id' => 1,
            ],
            [
                'name' => 'Problem Solving',
                'description' => 'Ability to identify, analyze, and solve complex problems',
                'category' => 'core',
                'skill_levels' => [
                    1 => 'Struggles with problem identification',
                    2 => 'Can solve simple problems with guidance',
                    3 => 'Solves problems independently',
                    4 => 'Excellent problem solver, helps others',
                    5 => 'Expert problem solver, creates innovative solutions'
                ],
                'organization_id' => 1,
            ],
            [
                'name' => 'Teamwork',
                'description' => 'Ability to work effectively as part of a team',
                'category' => 'behavioral',
                'skill_levels' => [
                    1 => 'Difficulty working in teams',
                    2 => 'Basic team collaboration',
                    3 => 'Good team player, contributes effectively',
                    4 => 'Excellent team member, facilitates collaboration',
                    5 => 'Outstanding team player, builds high-performing teams'
                ],
                'organization_id' => 1,
            ],
        ];

        foreach ($competencies as $competency) {
            PerformanceCompetency::create($competency);
        }

        // Create sample goals for employees
        $employees = Employee::take(3)->get();
        $users = User::take(3)->get();

        foreach ($employees as $index => $employee) {
            // Create individual goals
            PerformanceGoal::create([
                'employee_id' => $employee->id,
                'created_by' => $users->first()->id,
                'title' => 'Improve Technical Skills',
                'description' => 'Complete advanced training in relevant technologies and apply learnings to current projects',
                'type' => 'individual',
                'priority' => 'high',
                'start_date' => now()->startOfYear(),
                'target_date' => now()->endOfYear(),
                'status' => 'active',
                'weight_percentage' => 40,
                'target_value' => 3,
                'measurement_unit' => 'certifications',
                'success_criteria' => 'Complete 3 relevant certifications and demonstrate improved technical performance',
                'milestones' => [
                    ['title' => 'Complete first certification', 'date' => now()->addMonths(3)->format('Y-m-d'), 'description' => 'AWS or relevant certification'],
                    ['title' => 'Apply skills to project', 'date' => now()->addMonths(6)->format('Y-m-d'), 'description' => 'Implement new skills in current project'],
                    ['title' => 'Complete remaining certifications', 'date' => now()->addMonths(9)->format('Y-m-d'), 'description' => 'Complete 2 additional certifications'],
                ],
                'progress_percentage' => rand(20, 80),
                'organization_id' => 1,
            ]);

            PerformanceGoal::create([
                'employee_id' => $employee->id,
                'created_by' => $users->first()->id,
                'title' => 'Leadership Development',
                'description' => 'Develop leadership skills by mentoring junior team members and leading project initiatives',
                'type' => 'development',
                'priority' => 'medium',
                'start_date' => now()->startOfYear(),
                'target_date' => now()->endOfYear(),
                'status' => 'active',
                'weight_percentage' => 30,
                'target_value' => 2,
                'measurement_unit' => 'mentees',
                'success_criteria' => 'Successfully mentor 2 junior team members and lead at least 1 major project',
                'milestones' => [
                    ['title' => 'Identify mentees', 'date' => now()->addMonths(1)->format('Y-m-d'), 'description' => 'Select 2 junior team members to mentor'],
                    ['title' => 'Lead project initiative', 'date' => now()->addMonths(6)->format('Y-m-d'), 'description' => 'Take leadership role in major project'],
                ],
                'progress_percentage' => rand(30, 70),
                'organization_id' => 1,
            ]);

            // Create a self-assessment
            PerformanceSelfAssessment::create([
                'employee_id' => $employee->id,
                'assessment_date' => now()->subDays(rand(1, 30)),
                'period' => 'Q4',
                'achievements' => [
                    'Successfully completed major project ahead of schedule',
                    'Improved team collaboration and communication',
                    'Mentored 2 junior developers',
                    'Implemented new process that increased efficiency by 20%'
                ],
                'challenges_faced' => [
                    'Managing multiple priorities - created better time management system',
                    'Technical complexity in new project - sought additional training and mentorship',
                    'Team communication gaps - implemented regular check-ins and status updates'
                ],
                'strengths_identified' => 'Strong technical problem-solving abilities, good communication skills, ability to work under pressure, collaborative approach to teamwork',
                'areas_for_development' => 'Leadership skills, advanced technical certifications, project management methodologies, public speaking and presentation skills',
                'learning_goals' => [
                    'Complete AWS certification',
                    'Attend leadership development workshop',
                    'Learn advanced project management techniques',
                    'Improve public speaking skills'
                ],
                'career_aspirations' => [
                    'Move into technical leadership role within 2 years',
                    'Lead larger, more complex projects',
                    'Mentor and develop junior team members',
                    'Contribute to strategic technical decisions'
                ],
                'support_needed' => 'Access to leadership training programs, mentorship from senior leaders, opportunities to lead projects, budget for professional certifications',
                'overall_satisfaction' => rand(35, 50) / 10, // 3.5 to 5.0
                'additional_feedback' => 'Really enjoying the collaborative culture and growth opportunities. Would appreciate more challenging projects and leadership development opportunities.',
                'status' => 'submitted',
                'submitted_at' => now()->subDays(rand(1, 15)),
                'organization_id' => 1,
            ]);
        }

        $this->command->info('Performance enhancement data seeded successfully!');
    }
}