<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Employee;
use App\Models\Department;
use App\Models\Position;
use App\Models\LeaveType;
use App\Models\LeaveBalance;
use App\Models\Holiday;
use App\Models\Organization;
use Illuminate\Support\Facades\Hash;

class DatabaseSeeder extends Seeder
{
    public function run(): void
    {
        // Create default organization first (required for tenant scoping)
        $defaultOrg = Organization::firstOrCreate(
            ['slug' => 'default'],
            [
                'name' => 'Qaya Creatives',
                'status' => 'active',
                'settings' => [],
                'onboarding_step' => 'completed', // Mark onboarding as complete for demo
                'onboarding_completed_at' => now(), // Set completion timestamp
            ]
        );

        // Seed roles and permissions first
        $this->call(RolesAndPermissionsSeeder::class);

        // Create Departments
        $departments = [
            ['name' => 'Human Resources', 'color' => '#e74c3c', 'description' => 'HR Department', 'active' => true],
            ['name' => 'IT Department', 'color' => '#3498db', 'description' => 'Information Technology', 'active' => true],
            ['name' => 'Sales', 'color' => '#2ecc71', 'description' => 'Sales Department', 'active' => true],
            ['name' => 'Marketing', 'color' => '#f39c12', 'description' => 'Marketing Department', 'active' => true],
            ['name' => 'Finance', 'color' => '#9b59b6', 'description' => 'Finance Department', 'active' => true],
            ['name' => 'Operations', 'color' => '#1abc9c', 'description' => 'Operations Department', 'active' => true],
        ];

        foreach ($departments as $dept) {
            Department::create($dept);
        }

        // Create Positions
        $positions = [
            ['name' => 'CEO', 'department_id' => null, 'active' => true],
            ['name' => 'HR Manager', 'department_id' => 1, 'active' => true],
            ['name' => 'HR Executive', 'department_id' => 1, 'active' => true],
            ['name' => 'IT Manager', 'department_id' => 2, 'active' => true],
            ['name' => 'Software Developer', 'department_id' => 2, 'active' => true],
            ['name' => 'System Administrator', 'department_id' => 2, 'active' => true],
            ['name' => 'Sales Manager', 'department_id' => 3, 'active' => true],
            ['name' => 'Sales Executive', 'department_id' => 3, 'active' => true],
            ['name' => 'Marketing Manager', 'department_id' => 4, 'active' => true],
            ['name' => 'Marketing Executive', 'department_id' => 4, 'active' => true],
            ['name' => 'Finance Manager', 'department_id' => 5, 'active' => true],
            ['name' => 'Accountant', 'department_id' => 5, 'active' => true],
            ['name' => 'Operations Manager', 'department_id' => 6, 'active' => true],
        ];

        foreach ($positions as $pos) {
            Position::create($pos);
        }

        // Create Leave Types
        $leaveTypes = [
            ['name' => 'Annual Leave', 'days_allowed' => 21, 'is_paid' => true, 'color' => '#3498db', 'active' => true, 'description' => 'Annual vacation leave as per Kenya Employment Act'],
            ['name' => 'Sick Leave', 'days_allowed' => 7, 'is_paid' => true, 'color' => '#e74c3c', 'active' => true, 'description' => 'Medical leave for illness'],
            ['name' => 'Casual Leave', 'days_allowed' => 5, 'is_paid' => true, 'color' => '#2ecc71', 'active' => true, 'description' => 'Short-term personal leave'],
            ['name' => 'Maternity Leave', 'days_allowed' => 90, 'is_paid' => true, 'color' => '#f39c12', 'active' => true, 'description' => 'Maternity leave as per Kenya Employment Act (3 months)'],
            ['name' => 'Paternity Leave', 'days_allowed' => 14, 'is_paid' => true, 'color' => '#9b59b6', 'active' => true, 'description' => 'Paternity leave as per Kenya Employment Act (2 weeks)'],
            ['name' => 'Compassionate Leave', 'days_allowed' => 3, 'is_paid' => true, 'color' => '#34495e', 'active' => true, 'description' => 'Leave for bereavement or family emergencies'],
            ['name' => 'Study Leave', 'days_allowed' => 30, 'is_paid' => false, 'color' => '#16a085', 'active' => true, 'description' => 'Educational leave for studies'],
            ['name' => 'Sabbatical Leave', 'days_allowed' => 365, 'is_paid' => false, 'color' => '#8e44ad', 'active' => true, 'description' => 'Extended leave for personal development'],
            ['name' => 'Emergency Leave', 'days_allowed' => 2, 'is_paid' => true, 'color' => '#c0392b', 'active' => true, 'description' => 'Urgent personal matters'],
            ['name' => 'Unpaid Leave', 'days_allowed' => 0, 'is_paid' => false, 'color' => '#95a5a6', 'active' => true, 'description' => 'Leave without pay'],
        ];

        foreach ($leaveTypes as $type) {
            LeaveType::create($type);
        }

        // Create Holidays for Kenya (2026-2028)
        $holidays = [
            // 2026 holidays
            ['name' => 'New Year\'s Day', 'date' => '2026-01-01', 'year' => 2026, 'active' => true],
            ['name' => 'Eid al-Fitr', 'date' => '2026-03-20', 'year' => 2026, 'active' => true],
            ['name' => 'Good Friday', 'date' => '2026-04-03', 'year' => 2026, 'active' => true],
            ['name' => 'Easter Monday', 'date' => '2026-04-06', 'year' => 2026, 'active' => true],
            ['name' => 'Labour Day', 'date' => '2026-05-01', 'year' => 2026, 'active' => true],
            ['name' => 'Eid al-Adha', 'date' => '2026-05-27', 'year' => 2026, 'active' => true],
            ['name' => 'Madaraka Day', 'date' => '2026-06-01', 'year' => 2026, 'active' => true],
            ['name' => 'Mashujaa Day', 'date' => '2026-10-20', 'year' => 2026, 'active' => true],
            ['name' => 'Diwali', 'date' => '2026-11-08', 'year' => 2026, 'active' => true],
            ['name' => 'Jamhuri Day', 'date' => '2026-12-12', 'year' => 2026, 'active' => true],
            ['name' => 'Christmas Day', 'date' => '2026-12-25', 'year' => 2026, 'active' => true],
            ['name' => 'Boxing Day', 'date' => '2026-12-26', 'year' => 2026, 'active' => true],
            
            // 2027 holidays
            ['name' => 'New Year\'s Day', 'date' => '2027-01-01', 'year' => 2027, 'active' => true],
            ['name' => 'Eid al-Fitr', 'date' => '2027-03-09', 'year' => 2027, 'active' => true],
            ['name' => 'Good Friday', 'date' => '2027-03-26', 'year' => 2027, 'active' => true],
            ['name' => 'Easter Monday', 'date' => '2027-03-29', 'year' => 2027, 'active' => true],
            ['name' => 'Labour Day', 'date' => '2027-05-01', 'year' => 2027, 'active' => true],
            ['name' => 'Eid al-Adha', 'date' => '2027-05-16', 'year' => 2027, 'active' => true],
            ['name' => 'Madaraka Day', 'date' => '2027-06-01', 'year' => 2027, 'active' => true],
            ['name' => 'Mashujaa Day', 'date' => '2027-10-20', 'year' => 2027, 'active' => true],
            ['name' => 'Diwali', 'date' => '2027-10-29', 'year' => 2027, 'active' => true],
            ['name' => 'Jamhuri Day', 'date' => '2027-12-12', 'year' => 2027, 'active' => true],
            ['name' => 'Christmas Day', 'date' => '2027-12-25', 'year' => 2027, 'active' => true],
            ['name' => 'Boxing Day', 'date' => '2027-12-26', 'year' => 2027, 'active' => true],
            
            // 2028 holidays
            ['name' => 'New Year\'s Day', 'date' => '2028-01-01', 'year' => 2028, 'active' => true],
            ['name' => 'Eid al-Fitr', 'date' => '2028-02-26', 'year' => 2028, 'active' => true],
            ['name' => 'Good Friday', 'date' => '2028-04-14', 'year' => 2028, 'active' => true],
            ['name' => 'Easter Monday', 'date' => '2028-04-17', 'year' => 2028, 'active' => true],
            ['name' => 'Labour Day', 'date' => '2028-05-01', 'year' => 2028, 'active' => true],
            ['name' => 'Eid al-Adha', 'date' => '2028-05-05', 'year' => 2028, 'active' => true],
            ['name' => 'Madaraka Day', 'date' => '2028-06-01', 'year' => 2028, 'active' => true],
            ['name' => 'Mashujaa Day', 'date' => '2028-10-20', 'year' => 2028, 'active' => true],
            ['name' => 'Diwali', 'date' => '2028-11-17', 'year' => 2028, 'active' => true],
            ['name' => 'Jamhuri Day', 'date' => '2028-12-12', 'year' => 2028, 'active' => true],
            ['name' => 'Christmas Day', 'date' => '2028-12-25', 'year' => 2028, 'active' => true],
            ['name' => 'Boxing Day', 'date' => '2028-12-26', 'year' => 2028, 'active' => true],
        ];

        foreach ($holidays as $holiday) {
            Holiday::create($holiday);
        }

        // Create Admin User (verified by default for demo purposes)
        $orgAdmin = User::create([
            'name' => 'Admin User',
            'email' => 'qayacreatives@gmail.com',
            'password' => Hash::make('password'),
            'organization_id' => $defaultOrg->id, // Set organization
            'email_verified_at' => now(), // Mark as verified - no OTP needed
            'must_verify_email' => false, // Don't require OTP verification
            'first_login_completed' => true, // Mark first login as complete
        ]);

        // Assign admin role using Spatie
        $orgAdmin->assignRole('org_admin');

        $adminEmployee = Employee::create([
            'user_id' => $orgAdmin->id,
            'employee_code' => 'EMP00001',
            'first_name' => 'Admin',
            'last_name' => 'User',
            'email' => 'qayacreatives@gmail.com',
            'phone' => '+1234567890',
            'mobile' => '+1234567890',
            'date_of_birth' => '1990-01-01',
            'gender' => 'male',
            'marital_status' => 'single',
            'department_id' => 1,
            'position_id' => 1,
            'joining_date' => '2026-01-01',
            'work_email' => 'admin@company.com',
            'status' => 'active',
            'is_company_admin' => true,
        ]);

        // Create Sample Employees
        $sampleEmployees = [
            ['first_name' => 'John', 'last_name' => 'Doe', 'email' => 'john.doe@example.com', 'department_id' => 2, 'position_id' => 5, 'role' => 'employee'],
            ['first_name' => 'Jane', 'last_name' => 'Smith', 'email' => 'jane.smith@example.com', 'department_id' => 3, 'position_id' => 8, 'role' => 'manager'],
            ['first_name' => 'Michael', 'last_name' => 'Johnson', 'email' => 'michael.j@example.com', 'department_id' => 4, 'position_id' => 10, 'role' => 'hr_manager'],
            ['first_name' => 'Emily', 'last_name' => 'Williams', 'email' => 'emily.w@example.com', 'department_id' => 5, 'position_id' => 12, 'role' => 'accountant'],
            ['first_name' => 'David', 'last_name' => 'Brown', 'email' => 'david.b@example.com', 'department_id' => 6, 'position_id' => 13, 'role' => 'employee'],
        ];

        foreach ($sampleEmployees as $index => $empData) {
            $user = User::create([
                'name' => $empData['first_name'] . ' ' . $empData['last_name'],
                'email' => $empData['email'],
                'password' => Hash::make('password'),
                'organization_id' => $defaultOrg->id, // Set organization
                'email_verified_at' => now(), // Mark as verified for demo
                'must_verify_email' => false, // Don't require OTP verification
                'first_login_completed' => true, // Mark first login as complete
            ]);

            // Assign role using Spatie
            $user->assignRole($empData['role']);

            $employee = Employee::create([
                'user_id' => $user->id,
                'employee_code' => 'EMP' . str_pad($index + 2, 5, '0', STR_PAD_LEFT),
                'first_name' => $empData['first_name'],
                'last_name' => $empData['last_name'],
                'email' => $empData['email'],
                'phone' => '+1234567' . rand(100, 999),
                'mobile' => '+1234567' . rand(100, 999),
                'date_of_birth' => '199' . rand(0, 5) . '-0' . rand(1, 9) . '-' . rand(10, 28),
                'gender' => ['male', 'female'][rand(0, 1)],
                'marital_status' => ['single', 'married'][rand(0, 1)],
                'department_id' => $empData['department_id'],
                'position_id' => $empData['position_id'],
                'joining_date' => '2026-0' . rand(1, 9) . '-' . rand(10, 28),
                'work_email' => strtolower($empData['first_name']) . '@company.com',
                'status' => 'active',
            ]);

            // Create leave balances for each employee
            $leaveTypes = LeaveType::all();
            foreach ($leaveTypes as $leaveType) {
                LeaveBalance::create([
                    'employee_id' => $employee->id,
                    'leave_type_id' => $leaveType->id,
                    'year' => 2026,
                    'total_days' => $leaveType->days_allowed,
                    'used_days' => 0,
                    'remaining_days' => $leaveType->days_allowed,
                ]);
            }


            User::whereNull('organization_id')
                ->where('is_system_admin', false)
                ->where('id', '!=', $orgAdmin->id) // Don't update the org admin we just created
                ->update(['organization_id' => $defaultOrg->id]);
        }

        // Create leave balances for admin
        $leaveTypes = LeaveType::all();
        foreach ($leaveTypes as $leaveType) {
            LeaveBalance::create([
                'employee_id' => $adminEmployee->id,
                'leave_type_id' => $leaveType->id,
                'year' => 2026,
                'total_days' => $leaveType->days_allowed,
                'used_days' => 0,
                'remaining_days' => $leaveType->days_allowed,
                'organization_id' => 1,
            ]);
        }

        $this->command->info('Database seeded successfully!');
        $this->command->info('Admin Login: qayacreatives@gmail.com / password');
        
        // Run multitenancy data migration
        $this->call(MultitenancyDataSeeder::class);
        $this->call(RecruitmentPermissionsSeeder::class);
        $this->call(PerformanceEnhancementSeeder::class);
        $this->call(AnnouncementPermissionsSeeder::class);
        $this->call(TrainingCategoriesSeeder::class);
        $this->call(TrainingPermissionsSeeder::class);
        $this->call(TrainingTestDataSeeder::class);
        $this->call(ContractSeeder::class);
        $this->call(AttendanceSeeder::class);
        $this->call(AuditPermissionsSeeder::class);
        $this->call(PayrollSettingsSeeder::class);
        $this->call(ThemeSettingsSeeder::class);
        $this->call(SubscriptionPlansSeeder::class);
        $this->call(DemoSubscriptionSeeder::class);

    }
}
