<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // 1. Create subscription_plans table
        if (!Schema::hasTable('subscription_plans')) {
            Schema::create('subscription_plans', function (Blueprint $table) {
            $table->id();
            $table->string('name', 100); // Starter, Professional, Enterprise
            $table->string('slug', 100)->unique();
            $table->text('description')->nullable();
            $table->decimal('price_monthly', 10, 2); // KES price per month
            $table->decimal('price_annual', 10, 2)->nullable(); // KES price per year (with discount)
            
            // Limits and quotas
            $table->integer('max_employees')->nullable(); // null = unlimited
            $table->integer('max_users')->nullable();
            $table->integer('max_departments')->nullable();
            $table->integer('storage_limit_gb')->nullable();
            $table->integer('max_file_size_mb')->default(5);
            $table->integer('api_call_limit')->nullable(); // per month, null = no API access
            
            // Features (JSON)
            $table->json('features')->nullable(); // modules, support level, etc.
            
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            
            $table->index('slug');
            $table->index('is_active');
        });
        }

        // 2. Create organization_subscriptions table
        if (!Schema::hasTable('organization_subscriptions')) {
            Schema::create('organization_subscriptions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->foreignId('subscription_plan_id')->constrained()->onDelete('restrict');
            
            $table->enum('status', ['trial', 'active', 'suspended', 'cancelled', 'expired'])->default('trial');
            $table->enum('billing_cycle', ['monthly', 'annual'])->default('monthly');
            
            // Trial management
            $table->timestamp('trial_starts_at')->nullable();
            $table->timestamp('trial_ends_at')->nullable();
            $table->boolean('trial_extended')->default(false);
            
            // Billing period
            $table->timestamp('current_period_start')->nullable();
            $table->timestamp('current_period_end')->nullable();
            $table->timestamp('next_billing_date')->nullable();
            
            // Cancellation
            $table->boolean('cancel_at_period_end')->default(false);
            $table->timestamp('cancelled_at')->nullable();
            $table->text('cancellation_reason')->nullable();
            
            // Activation
            $table->timestamp('activated_at')->nullable();
            $table->timestamp('suspended_at')->nullable();
            $table->text('suspension_reason')->nullable();
            
            $table->timestamps();
            
            $table->index('organization_id');
            $table->index('status');
            $table->index(['organization_id', 'status']);
        });
        }

        // 3. Create invoices table
        if (!Schema::hasTable('invoices')) {
            Schema::create('invoices', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->foreignId('subscription_plan_id')->nullable()->constrained()->onDelete('set null');
            
            $table->string('invoice_number', 50)->unique();
            $table->enum('type', ['subscription', 'addon', 'overage', 'one_time'])->default('subscription');
            
            // Amounts in KES
            $table->decimal('subtotal', 10, 2);
            $table->decimal('tax', 10, 2)->default(0);
            $table->decimal('discount', 10, 2)->default(0);
            $table->decimal('total', 10, 2);
            
            $table->enum('status', ['draft', 'pending', 'paid', 'failed', 'refunded', 'cancelled'])->default('pending');
            
            // Dates
            $table->date('issue_date');
            $table->date('due_date');
            $table->timestamp('paid_at')->nullable();
            
            // Line items (JSON)
            $table->json('line_items')->nullable();
            
            // Notes
            $table->text('notes')->nullable();
            
            $table->timestamps();
            
            $table->index('organization_id');
            $table->index('invoice_number');
            $table->index('status');
            $table->index(['organization_id', 'status']);
        });
        }

        // 4. Create payments table (M-Pesa focused)
        if (!Schema::hasTable('payments')) {
            Schema::create('payments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('invoice_id')->nullable()->constrained()->onDelete('set null');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            
            $table->decimal('amount', 10, 2);
            $table->string('currency', 3)->default('KES');
            
            $table->enum('payment_method', ['mpesa', 'bank_transfer', 'cash', 'other'])->default('mpesa');
            $table->enum('status', ['pending', 'processing', 'completed', 'failed', 'refunded', 'cancelled'])->default('pending');
            
            // M-Pesa specific fields
            $table->string('mpesa_checkout_request_id')->nullable();
            $table->string('mpesa_merchant_request_id')->nullable();
            $table->string('mpesa_receipt_number')->nullable(); // Transaction ID from M-Pesa
            $table->string('mpesa_phone_number', 15)->nullable();
            $table->decimal('mpesa_amount', 10, 2)->nullable();
            $table->string('mpesa_result_code')->nullable();
            $table->text('mpesa_result_desc')->nullable();
            
            // Generic transaction fields
            $table->string('transaction_id', 100)->nullable()->unique();
            $table->string('reference_number', 100)->nullable();
            
            // Gateway response (full JSON)
            $table->json('gateway_response')->nullable();
            
            // Timestamps
            $table->timestamp('initiated_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamp('failed_at')->nullable();
            
            // Metadata
            $table->text('notes')->nullable();
            $table->string('ip_address', 45)->nullable();
            
            $table->timestamps();
            
            $table->index('organization_id');
            $table->index('invoice_id');
            $table->index('status');
            $table->index('transaction_id');
            $table->index('mpesa_checkout_request_id');
            $table->index('mpesa_receipt_number');
        });
        }

        // 5. Create organization_usage table (for tracking resource usage)
        if (!Schema::hasTable('organization_usage')) {
            Schema::create('organization_usage', function (Blueprint $table) {
            $table->id();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            
            $table->enum('metric_type', [
                'employees',
                'users',
                'departments',
                'storage_mb',
                'api_calls',
                'file_uploads'
            ]);
            
            $table->bigInteger('metric_value')->default(0);
            $table->date('recorded_date');
            $table->timestamp('recorded_at');
            
            $table->index('organization_id');
            $table->index(['organization_id', 'metric_type', 'recorded_date'], 'org_usage_org_metric_date_idx');
            $table->index('recorded_date');
        });
        }

        // 6. Update organizations table with subscription fields
        if (!Schema::hasColumn('organizations', 'subscription_status')) {
            Schema::table('organizations', function (Blueprint $table) {
            $table->enum('subscription_status', ['trial', 'active', 'suspended', 'cancelled', 'expired'])
                ->default('trial')
                ->after('status');
            $table->timestamp('trial_ends_at')->nullable()->after('subscription_status');
            $table->timestamp('subscription_ends_at')->nullable()->after('trial_ends_at');
        });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Drop in reverse order to handle foreign keys
        Schema::table('organizations', function (Blueprint $table) {
            $table->dropColumn(['subscription_status', 'trial_ends_at', 'subscription_ends_at']);
        });
        
        Schema::dropIfExists('organization_usage');
        Schema::dropIfExists('payments');
        Schema::dropIfExists('invoices');
        Schema::dropIfExists('organization_subscriptions');
        Schema::dropIfExists('subscription_plans');
    }
};
