<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('audit_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('user_id')->nullable()->constrained()->nullOnDelete();
            $table->foreignId('organization_id')->nullable()->constrained()->nullOnDelete();
            
            // For system admin actions, organization_id will be null
            // For organization user actions, organization_id will be set
            
            // Event details
            $table->string('event_type', 50); // create, update, delete, login, logout, etc.
            $table->string('auditable_type')->nullable(); // Model class name
            $table->unsignedBigInteger('auditable_id')->nullable(); // Model ID
            $table->string('description'); // Human readable description
            
            // Request context
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->string('url')->nullable();
            $table->string('method', 10)->nullable(); // GET, POST, PUT, DELETE
            
            // Data changes
            $table->json('old_values')->nullable(); // Previous values
            $table->json('new_values')->nullable(); // New values
            $table->json('metadata')->nullable(); // Additional context data
            
            // Status and categorization
            $table->enum('severity', ['low', 'medium', 'high', 'critical'])->default('low');
            $table->enum('category', [
                'authentication', 
                'authorization', 
                'data_access', 
                'data_modification', 
                'system_admin',
                'compliance',
                'security'
            ])->default('data_access');
            
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['user_id', 'created_at']);
            $table->index(['organization_id', 'created_at']);
            $table->index(['event_type', 'created_at']);
            $table->index(['auditable_type', 'auditable_id']);
            $table->index(['category', 'severity']);
            $table->index('created_at');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('audit_logs');
    }
};