<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ============================================
        // CONTRACTS ENHANCEMENTS
        // ============================================
        
        Schema::table('contracts', function (Blueprint $table) {
            // Employment type and payment structure
            $table->enum('employment_type', [
                'permanent', 'contract', 'casual', 'intern', 'freelance', 'part_time'
            ])->default('permanent')->after('contract_type');
            
            $table->enum('payment_frequency', [
                'monthly', 'weekly', 'daily', 'hourly', 'project_based'
            ])->default('monthly')->after('employment_type');
            
            // Rate structures
            $table->decimal('hourly_rate', 10, 2)->nullable()->after('salary');
            $table->decimal('daily_rate', 10, 2)->nullable()->after('hourly_rate');
            $table->decimal('overtime_rate_multiplier', 3, 2)->default(1.5)->after('daily_rate');
            
            // Tax and statutory deductions
            $table->boolean('withholding_tax_applicable')->default(false)->after('overtime_rate_multiplier');
            $table->decimal('withholding_tax_rate', 5, 4)->default(0.05)->after('withholding_tax_applicable');
            $table->boolean('statutory_deductions_applicable')->default(true)->after('withholding_tax_rate');
            
            // Working arrangements
            $table->decimal('standard_hours_per_day', 4, 2)->default(8.0)->after('working_hours_per_week');
            $table->integer('working_days_per_week')->default(5)->after('standard_hours_per_day');
            $table->json('working_schedule')->nullable()->after('working_days_per_week');
            
            // Contract financial terms
            $table->decimal('minimum_guaranteed_hours', 6, 2)->nullable()->after('working_schedule');
            $table->decimal('maximum_hours_per_week', 6, 2)->nullable()->after('minimum_guaranteed_hours');
            $table->boolean('overtime_eligible')->default(true)->after('maximum_hours_per_week');
            
            // Allowances and benefits eligibility
            $table->json('eligible_allowances')->nullable()->after('benefits');
            $table->json('eligible_statutory_benefits')->nullable()->after('eligible_allowances');
            
            // Performance and increment settings
            $table->boolean('performance_bonus_eligible')->default(true)->after('eligible_statutory_benefits');
            $table->boolean('annual_increment_eligible')->default(true)->after('performance_bonus_eligible');
            $table->decimal('increment_percentage', 5, 2)->nullable()->after('annual_increment_eligible');
            
            // Contract renewal and termination
            $table->integer('notice_period_days')->default(30)->after('increment_percentage');
            $table->boolean('auto_renewable')->default(false)->after('notice_period_days');
            $table->date('last_renewed_date')->nullable()->after('auto_renewable');
            
            // Indexes for performance
            $table->index(['employment_type', 'status']);
            $table->index(['payment_frequency', 'status']);
            $table->index(['end_date', 'status']);
        });

        // ============================================
        // PAYROLLS ENHANCEMENTS
        // ============================================
        
        Schema::table('payrolls', function (Blueprint $table) {
            // Employment type and contract reference
            $table->enum('employment_type', [
                'permanent', 'contract', 'casual', 'intern', 'freelance', 'part_time'
            ])->default('permanent')->after('employee_id');
            
            $table->foreignId('contract_id')->nullable()->constrained()->onDelete('set null')->after('employment_type');
            
            // Hours-based calculations
            $table->decimal('hourly_rate', 10, 2)->nullable()->after('basic_salary');
            $table->decimal('daily_rate', 10, 2)->nullable()->after('hourly_rate');
            $table->decimal('hours_worked', 6, 2)->default(0)->after('daily_rate');
            $table->decimal('regular_hours', 6, 2)->default(0)->after('hours_worked');
            $table->decimal('overtime_hours', 6, 2)->default(0)->after('regular_hours');
            $table->decimal('overtime_rate', 10, 2)->nullable()->after('overtime_hours');
            
            // Days-based calculations
            $table->integer('days_worked')->default(0)->after('overtime_rate');
            $table->integer('expected_working_days')->default(0)->after('days_worked');
            
            // Contract-specific pay components
            $table->decimal('hourly_pay', 12, 2)->default(0)->after('expected_working_days');
            $table->decimal('daily_pay', 12, 2)->default(0)->after('hourly_pay');
            $table->decimal('contract_basic_pay', 12, 2)->default(0)->after('daily_pay');
            
            // Withholding tax (for contractors)
            $table->decimal('withholding_tax', 10, 2)->default(0)->after('housing_levy');
            $table->decimal('withholding_tax_rate', 5, 4)->default(0)->after('withholding_tax');
            
            // Performance-based components
            $table->decimal('performance_bonus', 10, 2)->default(0)->after('bonuses');
            $table->decimal('merit_increase', 10, 2)->default(0)->after('performance_bonus');
            $table->decimal('commission', 10, 2)->default(0)->after('merit_increase');
            
            // Calculation metadata
            $table->json('calculation_breakdown')->nullable()->after('notes');
            $table->json('rate_details')->nullable()->after('calculation_breakdown');
            $table->boolean('prorated')->default(false)->after('rate_details');
            $table->text('proration_reason')->nullable()->after('prorated');
            
            // Approval and processing
            $table->enum('approval_status', ['pending', 'approved', 'rejected'])->default('pending')->after('status');
            $table->foreignId('approved_by')->nullable()->constrained('users')->onDelete('set null')->after('approval_status');
            $table->timestamp('approved_at')->nullable()->after('approved_by');
            $table->text('approval_notes')->nullable()->after('approved_at');
            
            // Indexes for performance
            $table->index(['employment_type', 'month', 'year']);
            $table->index(['contract_id', 'month', 'year']);
            $table->index(['approval_status', 'status']);
        });

        // ============================================
        // ATTENDANCES ENHANCEMENTS
        // ============================================
        
        Schema::table('attendances', function (Blueprint $table) {
            // Enhanced time tracking
            $table->decimal('regular_hours', 6, 2)->default(0)->after('worked_hours');
            $table->decimal('break_hours', 6, 2)->default(0)->after('regular_hours');
            $table->decimal('paid_break_hours', 6, 2)->default(0)->after('break_hours');
            
            // Overtime tracking
            $table->decimal('overtime_1_5x_hours', 6, 2)->default(0)->after('overtime_hours');
            $table->decimal('overtime_2x_hours', 6, 2)->default(0)->after('overtime_1_5x_hours');
            
            // Shift and premium time
            $table->enum('shift_type', ['day', 'night', 'weekend', 'holiday'])->default('day')->after('overtime_2x_hours');
            $table->decimal('night_shift_hours', 6, 2)->default(0)->after('shift_type');
            $table->decimal('weekend_hours', 6, 2)->default(0)->after('night_shift_hours');
            $table->decimal('holiday_hours', 6, 2)->default(0)->after('weekend_hours');
            
            // Payroll integration flags
            $table->boolean('included_in_payroll')->default(false)->after('holiday_hours');
            $table->foreignId('payroll_id')->nullable()->constrained()->onDelete('set null')->after('included_in_payroll');
            $table->date('payroll_period')->nullable()->after('payroll_id');
            
            // Late/Early tracking
            $table->integer('late_minutes')->default(0)->after('payroll_period');
            $table->integer('early_departure_minutes')->default(0)->after('late_minutes');
            $table->decimal('deduction_hours', 6, 2)->default(0)->after('early_departure_minutes');
            
            // Approval and verification
            $table->enum('approval_status', ['pending', 'approved', 'disputed'])->default('approved')->after('deduction_hours');
            $table->foreignId('approved_by')->nullable()->constrained('users')->onDelete('set null')->after('approval_status');
            $table->timestamp('approved_at')->nullable()->after('approved_by');
            
            // Location and device tracking
            $table->string('check_in_location')->nullable()->after('check_in_ip');
            $table->string('check_out_location')->nullable()->after('check_out_ip');
            $table->string('device_info')->nullable()->after('check_out_location');
            
            // Indexes for payroll queries
            $table->index(['employee_id', 'payroll_period']);
            $table->index(['included_in_payroll', 'payroll_period']);
            $table->index(['approval_status', 'date']);
        });
    }

    public function down(): void
    {
        // ============================================
        // ATTENDANCES ROLLBACK
        // ============================================
        
        Schema::table('attendances', function (Blueprint $table) {
            $table->dropForeign(['payroll_id']);
            $table->dropForeign(['approved_by']);
            
            $table->dropIndex(['employee_id', 'payroll_period']);
            $table->dropIndex(['included_in_payroll', 'payroll_period']);
            $table->dropIndex(['approval_status', 'date']);
            
            $table->dropColumn([
                'regular_hours',
                'break_hours',
                'paid_break_hours',
                'overtime_1_5x_hours',
                'overtime_2x_hours',
                'shift_type',
                'night_shift_hours',
                'weekend_hours',
                'holiday_hours',
                'included_in_payroll',
                'payroll_id',
                'payroll_period',
                'late_minutes',
                'early_departure_minutes',
                'deduction_hours',
                'approval_status',
                'approved_by',
                'approved_at',
                'check_in_location',
                'check_out_location',
                'device_info',
            ]);
        });

        // ============================================
        // PAYROLLS ROLLBACK
        // ============================================
        
        Schema::table('payrolls', function (Blueprint $table) {
            $table->dropForeign(['contract_id']);
            $table->dropForeign(['approved_by']);
            
            $table->dropIndex(['employment_type', 'month', 'year']);
            $table->dropIndex(['contract_id', 'month', 'year']);
            $table->dropIndex(['approval_status', 'status']);
            
            $table->dropColumn([
                'employment_type',
                'contract_id',
                'hourly_rate',
                'daily_rate',
                'hours_worked',
                'regular_hours',
                'overtime_hours',
                'overtime_rate',
                'days_worked',
                'expected_working_days',
                'hourly_pay',
                'daily_pay',
                'contract_basic_pay',
                'withholding_tax',
                'withholding_tax_rate',
                'performance_bonus',
                'merit_increase',
                'commission',
                'calculation_breakdown',
                'rate_details',
                'prorated',
                'proration_reason',
                'approval_status',
                'approved_by',
                'approved_at',
                'approval_notes',
            ]);
        });

        // ============================================
        // CONTRACTS ROLLBACK
        // ============================================
        
        Schema::table('contracts', function (Blueprint $table) {
            $table->dropIndex(['employment_type', 'status']);
            $table->dropIndex(['payment_frequency', 'status']);
            $table->dropIndex(['end_date', 'status']);
            
            $table->dropColumn([
                'employment_type',
                'payment_frequency',
                'hourly_rate',
                'daily_rate',
                'overtime_rate_multiplier',
                'withholding_tax_applicable',
                'withholding_tax_rate',
                'statutory_deductions_applicable',
                'standard_hours_per_day',
                'working_days_per_week',
                'working_schedule',
                'minimum_guaranteed_hours',
                'maximum_hours_per_week',
                'overtime_eligible',
                'eligible_allowances',
                'eligible_statutory_benefits',
                'performance_bonus_eligible',
                'annual_increment_eligible',
                'increment_percentage',
                'notice_period_days',
                'auto_renewable',
                'last_renewed_date',
            ]);
        });
    }
};
