<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // ============================================
        // FOUNDATION TABLES
        // ============================================
        
        // Training Categories
        Schema::create('training_categories', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->text('description')->nullable();
            $table->string('color')->default('#1f77b4');
            $table->boolean('is_active')->default(true);
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['organization_id', 'is_active']);
        });

        // Training Programs
        Schema::create('training_programs', function (Blueprint $table) {
            $table->id();
            $table->string('title');
            $table->text('description');
            $table->foreignId('training_category_id')->constrained()->onDelete('cascade');
            $table->enum('type', ['internal', 'external', 'online', 'workshop', 'seminar', 'certification']);
            $table->enum('delivery_method', ['in_person', 'virtual', 'hybrid', 'self_paced']);
            $table->integer('duration_hours');
            $table->decimal('cost_per_participant', 10, 2)->default(0);
            $table->integer('max_participants')->nullable();
            $table->json('prerequisites')->nullable();
            $table->json('learning_objectives')->nullable();
            $table->text('content_outline')->nullable();
            $table->string('instructor_name')->nullable();
            $table->string('instructor_email')->nullable();
            $table->string('provider_name')->nullable();
            $table->enum('difficulty_level', ['beginner', 'intermediate', 'advanced']);
            $table->json('target_roles')->nullable();
            $table->boolean('is_mandatory')->default(false);
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['organization_id', 'is_active']);
            $table->index(['training_category_id', 'type']);
        });

        // Training Sessions
        Schema::create('training_sessions', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->string('session_code')->unique();
            $table->datetime('start_datetime');
            $table->datetime('end_datetime');
            $table->string('location')->nullable();
            $table->string('virtual_meeting_link')->nullable();
            $table->text('session_notes')->nullable();
            $table->integer('max_participants')->nullable();
            $table->enum('status', ['scheduled', 'in_progress', 'completed', 'cancelled'])->default('scheduled');
            $table->foreignId('instructor_id')->nullable()->constrained('users')->onDelete('set null');
            $table->decimal('actual_cost', 10, 2)->default(0);
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['training_program_id', 'status']);
            $table->index(['start_datetime', 'status']);
        });

        // Training Enrollments
        Schema::create('training_enrollments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_session_id')->constrained()->onDelete('cascade');
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->enum('enrollment_type', ['mandatory', 'voluntary', 'manager_assigned']);
            $table->enum('status', ['enrolled', 'attended', 'completed', 'failed', 'cancelled', 'no_show']);
            $table->datetime('enrolled_at');
            $table->datetime('attended_at')->nullable();
            $table->datetime('completed_at')->nullable();
            $table->decimal('completion_score', 5, 2)->nullable();
            $table->text('feedback')->nullable();
            $table->integer('rating')->nullable();
            $table->text('notes')->nullable();
            $table->foreignId('enrolled_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();

            $table->unique(['training_session_id', 'employee_id']);
            $table->index(['employee_id', 'status']);
            $table->index(['training_session_id', 'status']);
        });

        // Training Materials
        Schema::create('training_materials', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->string('title');
            $table->text('description')->nullable();
            $table->enum('type', ['document', 'video', 'presentation', 'link', 'quiz', 'assignment']);
            $table->string('file_path')->nullable();
            $table->string('external_url')->nullable();
            $table->integer('order_index')->default(0);
            $table->boolean('is_required')->default(false);
            $table->boolean('is_downloadable')->default(true);
            $table->timestamps();

            $table->index(['training_program_id', 'order_index']);
        });

        // ============================================
        // SKILLS INTEGRATION
        // ============================================
        
        // Training Skills Mapping
        Schema::create('training_program_skills', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->foreignId('skill_id')->constrained()->onDelete('cascade');
            $table->integer('skill_level_improvement')->default(1);
            $table->boolean('is_primary_skill')->default(false);
            $table->timestamps();

            $table->unique(['training_program_id', 'skill_id']);
        });

        // Training Requirements
        Schema::create('training_requirements', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->string('requirement_type');
            $table->unsignedBigInteger('requirement_target_id')->nullable();
            $table->integer('required_within_days')->default(90);
            $table->integer('renewal_period_months')->nullable();
            $table->date('effective_from');
            $table->date('effective_until')->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->index(['requirement_type', 'requirement_target_id'], 'training_req_type_target_idx');
            $table->index(['effective_from', 'effective_until'], 'training_req_dates_idx');
        });

        // Training Assessments
        Schema::create('training_assessments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->string('title');
            $table->text('description')->nullable();
            $table->enum('type', ['quiz', 'assignment', 'practical', 'presentation']);
            $table->integer('passing_score')->default(70);
            $table->integer('max_attempts')->default(3);
            $table->integer('time_limit_minutes')->nullable();
            $table->json('questions')->nullable();
            $table->boolean('is_required')->default(true);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        // Assessment Attempts
        Schema::create('training_assessment_attempts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_assessment_id')->constrained()->onDelete('cascade');
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('training_enrollment_id')->constrained()->onDelete('cascade');
            $table->integer('attempt_number');
            $table->datetime('started_at');
            $table->datetime('completed_at')->nullable();
            $table->json('answers')->nullable();
            $table->decimal('score', 5, 2)->nullable();
            $table->boolean('passed')->default(false);
            $table->text('feedback')->nullable();
            $table->timestamps();

            $table->index(['training_assessment_id', 'employee_id'], 'training_attempt_assessment_emp_idx');
        });

        // Training Certificates
        Schema::create('training_certificates', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_enrollment_id')->constrained()->onDelete('cascade');
            $table->string('certificate_number')->unique();
            $table->date('issued_date');
            $table->date('expiry_date')->nullable();
            $table->string('certificate_file_path')->nullable();
            $table->json('certificate_data')->nullable();
            $table->boolean('is_valid')->default(true);
            $table->foreignId('issued_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();

            $table->index(['issued_date', 'expiry_date']);
        });

        // Training Evaluations
        Schema::create('training_evaluations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_session_id')->constrained()->onDelete('cascade');
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->integer('content_rating');
            $table->integer('instructor_rating');
            $table->integer('delivery_rating');
            $table->integer('relevance_rating');
            $table->integer('overall_rating');
            $table->text('what_worked_well')->nullable();
            $table->text('areas_for_improvement')->nullable();
            $table->text('additional_comments')->nullable();
            $table->boolean('would_recommend')->default(true);
            $table->datetime('submitted_at');
            $table->timestamps();

            $table->unique(['training_session_id', 'employee_id'], 'training_eval_session_emp_unique');
        });

        // Training Budget Tracking
        Schema::create('training_budgets', function (Blueprint $table) {
            $table->id();
            $table->foreignId('department_id')->nullable()->constrained()->onDelete('cascade');
            $table->integer('year');
            $table->decimal('allocated_budget', 12, 2);
            $table->decimal('spent_budget', 12, 2)->default(0);
            $table->decimal('committed_budget', 12, 2)->default(0);
            $table->text('notes')->nullable();
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();

            $table->unique(['department_id', 'year', 'organization_id']);
        });

        // ============================================
        // LMS FEATURES
        // ============================================
        
        // Learning Paths
        Schema::create('learning_paths', function (Blueprint $table) {
            $table->id();
            $table->string('title');
            $table->text('description');
            $table->string('image_path')->nullable();
            $table->enum('difficulty_level', ['beginner', 'intermediate', 'advanced']);
            $table->integer('estimated_duration_hours');
            $table->json('target_roles')->nullable();
            $table->json('learning_outcomes')->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('created_by')->constrained('users')->onDelete('cascade');
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();
        });

        // Learning Path Steps
        Schema::create('learning_path_steps', function (Blueprint $table) {
            $table->id();
            $table->foreignId('learning_path_id')->constrained()->onDelete('cascade');
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->integer('step_order');
            $table->boolean('is_required')->default(true);
            $table->json('prerequisites')->nullable();
            $table->timestamps();

            $table->unique(['learning_path_id', 'training_program_id']);
            $table->index(['learning_path_id', 'step_order']);
        });

        // Learning Path Enrollments
        Schema::create('learning_path_enrollments', function (Blueprint $table) {
            $table->id();
            $table->foreignId('learning_path_id')->constrained()->onDelete('cascade');
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->enum('status', ['enrolled', 'in_progress', 'completed', 'paused', 'cancelled']);
            $table->datetime('enrolled_at');
            $table->datetime('started_at')->nullable();
            $table->datetime('completed_at')->nullable();
            $table->decimal('progress_percentage', 5, 2)->default(0);
            $table->integer('current_step')->default(1);
            $table->foreignId('enrolled_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();

            $table->unique(['learning_path_id', 'employee_id']);
        });

        // Training Performance Correlation
        Schema::create('training_performance_impacts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('performance_review_id')->nullable()->constrained()->onDelete('set null');
            $table->decimal('performance_score_before', 5, 2)->nullable();
            $table->decimal('performance_score_after', 5, 2)->nullable();
            $table->date('training_completed_date');
            $table->date('performance_measured_date');
            $table->json('skill_improvements')->nullable();
            $table->text('impact_notes')->nullable();
            $table->foreignId('measured_by')->constrained('users')->onDelete('cascade');
            $table->timestamps();

            $table->index(['training_program_id', 'training_completed_date'], 'training_perf_program_date_idx');
        });

        // Training Recommendations Engine
        Schema::create('training_recommendations', function (Blueprint $table) {
            $table->id();
            $table->foreignId('employee_id')->constrained()->onDelete('cascade');
            $table->foreignId('training_program_id')->constrained()->onDelete('cascade');
            $table->enum('recommendation_type', ['skill_gap', 'career_development', 'performance_improvement', 'mandatory', 'peer_suggestion']);
            $table->text('reason');
            $table->integer('priority_score')->default(50);
            $table->json('supporting_data')->nullable();
            $table->enum('status', ['pending', 'accepted', 'declined', 'completed']);
            $table->datetime('recommended_at');
            $table->datetime('responded_at')->nullable();
            $table->foreignId('recommended_by')->nullable()->constrained('users')->onDelete('set null');
            $table->timestamps();

            $table->index(['employee_id', 'status']);
            $table->index(['recommendation_type', 'priority_score'], 'training_rec_type_priority_idx');
        });

        // External LMS Integration
        Schema::create('external_lms_integrations', function (Blueprint $table) {
            $table->id();
            $table->string('provider_name');
            $table->string('api_endpoint');
            $table->json('api_credentials');
            $table->json('sync_settings');
            $table->datetime('last_sync_at')->nullable();
            $table->boolean('is_active')->default(true);
            $table->foreignId('organization_id')->constrained()->onDelete('cascade');
            $table->timestamps();
        });

        // External Course Sync
        Schema::create('external_training_courses', function (Blueprint $table) {
            $table->id();
            $table->foreignId('external_lms_integration_id')->constrained()->onDelete('cascade');
            $table->string('external_course_id');
            $table->foreignId('training_program_id')->nullable()->constrained()->onDelete('set null');
            $table->string('course_title');
            $table->text('course_description')->nullable();
            $table->string('course_url');
            $table->integer('duration_hours')->nullable();
            $table->decimal('cost', 10, 2)->nullable();
            $table->json('course_metadata')->nullable();
            $table->datetime('last_synced_at');
            $table->timestamps();

            $table->unique(['external_lms_integration_id', 'external_course_id'], 'ext_lms_course_unique');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('external_training_courses');
        Schema::dropIfExists('external_lms_integrations');
        Schema::dropIfExists('training_recommendations');
        Schema::dropIfExists('training_performance_impacts');
        Schema::dropIfExists('learning_path_enrollments');
        Schema::dropIfExists('learning_path_steps');
        Schema::dropIfExists('learning_paths');
        Schema::dropIfExists('training_budgets');
        Schema::dropIfExists('training_evaluations');
        Schema::dropIfExists('training_certificates');
        Schema::dropIfExists('training_assessment_attempts');
        Schema::dropIfExists('training_assessments');
        Schema::dropIfExists('training_requirements');
        Schema::dropIfExists('training_program_skills');
        Schema::dropIfExists('training_materials');
        Schema::dropIfExists('training_enrollments');
        Schema::dropIfExists('training_sessions');
        Schema::dropIfExists('training_programs');
        Schema::dropIfExists('training_categories');
    }
};
