<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Create organizations table
        Schema::create('organizations', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('slug')->unique();
            $table->string('logo')->nullable();
            $table->json('settings')->nullable();
            $table->enum('status', ['active', 'inactive', 'suspended'])->default('active');
            $table->timestamps();
        });

        // Add organization context to users
        Schema::table('users', function (Blueprint $table) {
            $table->foreignId('organization_id')->nullable()->constrained()->onDelete('cascade');
            $table->boolean('is_system_admin')->default(false);
        });

        // Add organization_id to business tables
        $tables = [
            'employees',
            'departments', 
            'positions',
            'attendances',
            'attendance_settings',
            'leave_requests',
            'leave_balances',
            'leave_types',
            'payrolls',
            'performance_reviews',
            'holidays',
            'announcements',
            'contracts',
            'p9_forms',
            'employee_documents',
            'documents',
            'skills',
        ];

        foreach ($tables as $table) {
            if (Schema::hasTable($table)) {
                Schema::table($table, function (Blueprint $table) {
                    $table->foreignId('organization_id')->nullable()->constrained()->onDelete('cascade');
                });
            }
        }

        // Add indexes for organization_id on all tenant tables
        foreach ($tables as $tableName) {
            if (Schema::hasTable($tableName) && Schema::hasColumn($tableName, 'organization_id')) {
                Schema::table($tableName, function (Blueprint $table) use ($tableName) {
                    $table->index('organization_id', "idx_{$tableName}_organization_id");
                });
            }
        }

        // Add composite indexes for common query patterns
        if (Schema::hasTable('employees')) {
            Schema::table('employees', function (Blueprint $table) {
                $table->index(['organization_id', 'status'], 'idx_employees_org_status');
                $table->index(['organization_id', 'department_id'], 'idx_employees_org_dept');
            });
        }

        if (Schema::hasTable('attendances')) {
            Schema::table('attendances', function (Blueprint $table) {
                $table->index(['organization_id', 'date'], 'idx_attendances_org_date');
                $table->index(['organization_id', 'employee_id', 'date'], 'idx_attendances_org_emp_date');
            });
        }

        if (Schema::hasTable('leave_requests')) {
            Schema::table('leave_requests', function (Blueprint $table) {
                $table->index(['organization_id', 'status'], 'idx_leave_requests_org_status');
                $table->index(['organization_id', 'start_date'], 'idx_leave_requests_org_start');
            });
        }

        if (Schema::hasTable('payrolls')) {
            Schema::table('payrolls', function (Blueprint $table) {
                $table->index(['organization_id', 'year', 'month'], 'idx_payrolls_org_year_month');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        $tables = [
            'employees',
            'departments', 
            'positions',
            'attendances',
            'attendance_settings',
            'leave_requests',
            'leave_balances',
            'leave_types',
            'payrolls',
            'performance_reviews',
            'holidays',
            'announcements',
            'contracts',
            'p9_forms',
            'employee_documents',
            'documents',
            'skills',
        ];

        // Drop composite indexes
        if (Schema::hasTable('employees')) {
            Schema::table('employees', function (Blueprint $table) {
                $table->dropIndex('idx_employees_org_status');
                $table->dropIndex('idx_employees_org_dept');
            });
        }

        if (Schema::hasTable('attendances')) {
            Schema::table('attendances', function (Blueprint $table) {
                $table->dropIndex('idx_attendances_org_date');
                $table->dropIndex('idx_attendances_org_emp_date');
            });
        }

        if (Schema::hasTable('leave_requests')) {
            Schema::table('leave_requests', function (Blueprint $table) {
                $table->dropIndex('idx_leave_requests_org_status');
                $table->dropIndex('idx_leave_requests_org_start');
            });
        }

        if (Schema::hasTable('payrolls')) {
            Schema::table('payrolls', function (Blueprint $table) {
                $table->dropIndex('idx_payrolls_org_year_month');
            });
        }

        // Drop organization_id indexes
        foreach ($tables as $tableName) {
            if (Schema::hasTable($tableName)) {
                Schema::table($tableName, function (Blueprint $table) use ($tableName) {
                    $table->dropIndex("idx_{$tableName}_organization_id");
                });
            }
        }

        // Drop organization_id columns from business tables
        foreach ($tables as $table) {
            if (Schema::hasTable($table)) {
                Schema::table($table, function (Blueprint $table) {
                    $table->dropForeign(['organization_id']);
                    $table->dropColumn('organization_id');
                });
            }
        }

        // Drop organization context from users
        Schema::table('users', function (Blueprint $table) {
            $table->dropForeign(['organization_id']);
            $table->dropColumn(['organization_id', 'is_system_admin']);
        });

        // Drop organizations table
        Schema::dropIfExists('organizations');
    }
};
