<?php

namespace App\Services;

use App\Models\Organization;
use App\Models\OrganizationUsage;
use App\Models\Employee;
use App\Models\User;
use App\Models\Department;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class UsageTrackingService
{
    /**
     * Track current usage for organization
     *
     * @param Organization $organization
     * @return array
     */
    public function trackUsage(Organization $organization): array
    {
        $metrics = [
            'employees' => $this->getEmployeeCount($organization),
            'users' => $this->getUserCount($organization),
            'departments' => $this->getDepartmentCount($organization),
            'storage_mb' => $this->getStorageUsage($organization),
        ];

        // Record each metric
        foreach ($metrics as $type => $value) {
            OrganizationUsage::recordMetric($organization->id, $type, $value);
        }

        return $metrics;
    }

    /**
     * Get employee count
     *
     * @param Organization $organization
     * @return int
     */
    public function getEmployeeCount(Organization $organization): int
    {
        return Employee::where('organization_id', $organization->id)
            ->where('status', 'active')
            ->count();
    }

    /**
     * Get user count
     *
     * @param Organization $organization
     * @return int
     */
    public function getUserCount(Organization $organization): int
    {
        return User::where('organization_id', $organization->id)->count();
    }

    /**
     * Get department count
     *
     * @param Organization $organization
     * @return int
     */
    public function getDepartmentCount(Organization $organization): int
    {
        return Department::where('organization_id', $organization->id)
            ->where('active', true)
            ->count();
    }

    /**
     * Get storage usage in MB
     *
     * @param Organization $organization
     * @return int
     */
    public function getStorageUsage(Organization $organization): int
    {
        // Calculate storage from employee documents and photos
        $storage = DB::table('employee_documents')
            ->join('employees', 'employee_documents.employee_id', '=', 'employees.id')
            ->where('employees.organization_id', $organization->id)
            ->sum('employee_documents.file_size') ?? 0;

        // Add employee photos
        $photoStorage = Employee::where('organization_id', $organization->id)
            ->whereNotNull('photo')
            ->get()
            ->sum(function ($employee) {
                $path = storage_path('app/public/' . $employee->photo);
                return file_exists($path) ? filesize($path) : 0;
            });

        $totalBytes = $storage + $photoStorage;
        return (int) ceil($totalBytes / 1024 / 1024); // Convert to MB
    }

    /**
     * Check if organization has exceeded quota
     *
     * @param Organization $organization
     * @param string $metric
     * @return array
     */
    public function checkQuota(Organization $organization, string $metric): array
    {
        $plan = $organization->currentPlan();
        
        if (!$plan) {
            return [
                'exceeded' => false,
                'current' => 0,
                'limit' => 0,
                'percentage' => 0,
            ];
        }

        $limit = $organization->getQuotaLimit($metric);
        $current = $this->getCurrentUsage($organization, $metric);

        // null limit means unlimited
        if (is_null($limit)) {
            return [
                'exceeded' => false,
                'current' => $current,
                'limit' => null,
                'percentage' => 0,
                'unlimited' => true,
            ];
        }

        $exceeded = $current >= $limit;
        $percentage = $limit > 0 ? ($current / $limit) * 100 : 0;

        return [
            'exceeded' => $exceeded,
            'current' => $current,
            'limit' => $limit,
            'percentage' => round($percentage, 2),
            'remaining' => max(0, $limit - $current),
            'unlimited' => false,
        ];
    }

    /**
     * Get current usage for metric
     *
     * @param Organization $organization
     * @param string $metric
     * @return int
     */
    protected function getCurrentUsage(Organization $organization, string $metric): int
    {
        return match($metric) {
            'employees' => $this->getEmployeeCount($organization),
            'users' => $this->getUserCount($organization),
            'departments' => $this->getDepartmentCount($organization),
            'storage' => $this->getStorageUsage($organization),
            default => 0,
        };
    }

    /**
     * Get usage summary for organization
     *
     * @param Organization $organization
     * @return array
     */
    public function getUsageSummary(Organization $organization): array
    {
        return [
            'employees' => $this->checkQuota($organization, 'employees'),
            'users' => $this->checkQuota($organization, 'users'),
            'departments' => $this->checkQuota($organization, 'departments'),
            'storage' => $this->checkQuota($organization, 'storage'),
        ];
    }

    /**
     * Check if organization can add more of a resource
     *
     * @param Organization $organization
     * @param string $metric
     * @param int $count
     * @return bool
     */
    public function canAdd(Organization $organization, string $metric, int $count = 1): bool
    {
        $quota = $this->checkQuota($organization, $metric);
        
        if ($quota['unlimited'] ?? false) {
            return true;
        }

        return ($quota['current'] + $count) <= $quota['limit'];
    }

    /**
     * Get usage history for organization
     *
     * @param Organization $organization
     * @param string $metric
     * @param int $days
     * @return array
     */
    public function getUsageHistory(Organization $organization, string $metric, int $days = 30): array
    {
        $startDate = now()->subDays($days);
        
        $usage = OrganizationUsage::where('organization_id', $organization->id)
            ->where('metric_type', $metric)
            ->where('recorded_date', '>=', $startDate)
            ->orderBy('recorded_date')
            ->get()
            ->groupBy(function ($item) {
                return $item->recorded_date->format('Y-m-d');
            })
            ->map(function ($group) {
                return $group->last()->metric_value;
            });

        return $usage->toArray();
    }
}
