<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Str;

class TwoFactorAuthService
{
    /**
     * Generate a new 2FA secret for the user
     */
    public function generateSecret(): string
    {
        return strtoupper(Str::random(32));
    }

    /**
     * Generate recovery codes for the user
     */
    public function generateRecoveryCodes(): array
    {
        $codes = [];
        for ($i = 0; $i < 8; $i++) {
            $codes[] = strtoupper(Str::random(10));
        }
        return $codes;
    }

    /**
     * Enable 2FA for a user
     */
    public function enableTwoFactor(User $user): array
    {
        $secret = $this->generateSecret();
        $recoveryCodes = $this->generateRecoveryCodes();

        $user->update([
            'two_factor_secret' => Crypt::encryptString($secret),
            'two_factor_recovery_codes' => $recoveryCodes,
            'two_factor_confirmed_at' => now(),
        ]);

        return [
            'secret' => $secret,
            'recovery_codes' => $recoveryCodes,
            'qr_code_url' => $this->generateQrCodeUrl($user, $secret)
        ];
    }

    /**
     * Disable 2FA for a user
     */
    public function disableTwoFactor(User $user): void
    {
        $user->update([
            'two_factor_secret' => null,
            'two_factor_recovery_codes' => null,
            'two_factor_confirmed_at' => null,
        ]);
    }

    /**
     * Check if user has 2FA enabled
     */
    public function isEnabled(User $user): bool
    {
        return !is_null($user->two_factor_secret) && !is_null($user->two_factor_confirmed_at);
    }

    /**
     * Verify a 2FA code
     */
    public function verifyCode(User $user, string $code): bool
    {
        if (!$this->isEnabled($user)) {
            return false;
        }

        // Check if it's a recovery code
        if ($this->verifyRecoveryCode($user, $code)) {
            return true;
        }

        // For demo purposes, we'll accept a simple time-based code
        // In production, you'd use a proper TOTP library
        $secret = Crypt::decryptString($user->two_factor_secret);
        $expectedCode = $this->generateTimeBasedCode($secret);
        
        return hash_equals($expectedCode, $code);
    }

    /**
     * Verify a recovery code
     */
    public function verifyRecoveryCode(User $user, string $code): bool
    {
        $recoveryCodes = $user->two_factor_recovery_codes ?? [];
        
        if (in_array(strtoupper($code), $recoveryCodes)) {
            // Remove the used recovery code
            $recoveryCodes = array_diff($recoveryCodes, [strtoupper($code)]);
            $user->update(['two_factor_recovery_codes' => array_values($recoveryCodes)]);
            return true;
        }

        return false;
    }

    /**
     * Generate a simple time-based code (for demo purposes)
     */
    private function generateTimeBasedCode(string $secret): string
    {
        $timeSlice = floor(time() / 30);
        return substr(hash('sha256', $secret . $timeSlice), 0, 6);
    }

    /**
     * Generate QR code URL for Google Authenticator
     */
    private function generateQrCodeUrl(User $user, string $secret): string
    {
        $appName = config('app.name', 'Laravel');
        $email = $user->email;
        
        $otpAuthUrl = "otpauth://totp/{$appName}:{$email}?secret={$secret}&issuer={$appName}";
        
        return "https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=" . urlencode($otpAuthUrl);
    }
}