<?php

namespace App\Services;

use App\Models\Organization;
use App\Models\OrganizationSubscription;
use App\Models\SubscriptionPlan;
use App\Models\Invoice;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class SubscriptionService
{
    /**
     * Create a new subscription for organization
     *
     * @param Organization $organization
     * @param SubscriptionPlan $plan
     * @param string $billingCycle
     * @param bool $startTrial
     * @return OrganizationSubscription
     */
    public function createSubscription(
        Organization $organization,
        SubscriptionPlan $plan,
        string $billingCycle = 'monthly',
        bool $startTrial = true
    ): OrganizationSubscription {
        DB::beginTransaction();
        
        try {
            $now = now();
            $trialDays = config('subscription.trial_days', 14);
            
            $subscription = OrganizationSubscription::create([
                'organization_id' => $organization->id,
                'subscription_plan_id' => $plan->id,
                'status' => $startTrial ? 'trial' : 'active',
                'billing_cycle' => $billingCycle,
                'trial_starts_at' => $startTrial ? $now : null,
                'trial_ends_at' => $startTrial ? $now->copy()->addDays($trialDays) : null,
                'current_period_start' => $now,
                'current_period_end' => $this->calculatePeriodEnd($now, $billingCycle),
                'next_billing_date' => $startTrial ? 
                    $now->copy()->addDays($trialDays) : 
                    $this->calculatePeriodEnd($now, $billingCycle),
                'activated_at' => $startTrial ? null : $now,
            ]);

            // Update organization
            $organization->update([
                'subscription_status' => $startTrial ? 'trial' : 'active',
                'trial_ends_at' => $subscription->trial_ends_at,
                'subscription_ends_at' => $subscription->current_period_end,
            ]);

            Log::info('Subscription created', [
                'organization_id' => $organization->id,
                'plan' => $plan->name,
                'status' => $subscription->status,
            ]);

            DB::commit();
            return $subscription;

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed to create subscription', [
                'organization_id' => $organization->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Upgrade or downgrade subscription
     *
     * @param OrganizationSubscription $subscription
     * @param SubscriptionPlan $newPlan
     * @param bool $immediate
     * @return OrganizationSubscription
     */
    public function changePlan(
        OrganizationSubscription $subscription,
        SubscriptionPlan $newPlan,
        bool $immediate = true
    ): OrganizationSubscription {
        DB::beginTransaction();
        
        try {
            $oldPlan = $subscription->plan;
            
            if ($immediate) {
                // Immediate change
                $subscription->update([
                    'subscription_plan_id' => $newPlan->id,
                ]);

                Log::info('Subscription plan changed immediately', [
                    'organization_id' => $subscription->organization_id,
                    'old_plan' => $oldPlan->name,
                    'new_plan' => $newPlan->name,
                ]);
            } else {
                // Change at end of billing period
                $subscription->update([
                    'subscription_plan_id' => $newPlan->id,
                    'cancel_at_period_end' => false, // Reset if was cancelling
                ]);

                Log::info('Subscription plan scheduled to change', [
                    'organization_id' => $subscription->organization_id,
                    'old_plan' => $oldPlan->name,
                    'new_plan' => $newPlan->name,
                    'effective_date' => $subscription->current_period_end,
                ]);
            }

            DB::commit();
            return $subscription->fresh();

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed to change subscription plan', [
                'subscription_id' => $subscription->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Convert trial to paid subscription
     *
     * @param OrganizationSubscription $subscription
     * @return OrganizationSubscription
     */
    public function convertTrialToPaid(OrganizationSubscription $subscription): OrganizationSubscription
    {
        if ($subscription->status !== 'trial') {
            throw new Exception('Subscription is not on trial');
        }

        DB::beginTransaction();
        
        try {
            $now = now();
            
            $subscription->update([
                'status' => 'active',
                'activated_at' => $now,
                'current_period_start' => $now,
                'current_period_end' => $this->calculatePeriodEnd($now, $subscription->billing_cycle),
                'next_billing_date' => $this->calculatePeriodEnd($now, $subscription->billing_cycle),
            ]);

            $subscription->organization->update([
                'subscription_status' => 'active',
                'subscription_ends_at' => $subscription->current_period_end,
            ]);

            Log::info('Trial converted to paid subscription', [
                'organization_id' => $subscription->organization_id,
                'subscription_id' => $subscription->id,
            ]);

            DB::commit();
            return $subscription->fresh();

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed to convert trial', [
                'subscription_id' => $subscription->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Extend trial period
     *
     * @param OrganizationSubscription $subscription
     * @param int $days
     * @return OrganizationSubscription
     */
    public function extendTrial(OrganizationSubscription $subscription, int $days = 7): OrganizationSubscription
    {
        if ($subscription->status !== 'trial') {
            throw new Exception('Subscription is not on trial');
        }

        if ($subscription->trial_extended) {
            throw new Exception('Trial has already been extended');
        }

        $subscription->update([
            'trial_ends_at' => $subscription->trial_ends_at->addDays($days),
            'trial_extended' => true,
            'next_billing_date' => $subscription->trial_ends_at->addDays($days),
        ]);

        $subscription->organization->update([
            'trial_ends_at' => $subscription->trial_ends_at,
        ]);

        Log::info('Trial extended', [
            'organization_id' => $subscription->organization_id,
            'days' => $days,
            'new_end_date' => $subscription->trial_ends_at,
        ]);

        return $subscription->fresh();
    }

    /**
     * Cancel subscription
     *
     * @param OrganizationSubscription $subscription
     * @param string|null $reason
     * @param bool $immediately
     * @return OrganizationSubscription
     */
    public function cancelSubscription(
        OrganizationSubscription $subscription,
        ?string $reason = null,
        bool $immediately = false
    ): OrganizationSubscription {
        if ($immediately) {
            $subscription->update([
                'status' => 'cancelled',
                'cancelled_at' => now(),
                'cancellation_reason' => $reason,
            ]);

            $subscription->organization->update([
                'subscription_status' => 'cancelled',
            ]);

            Log::info('Subscription cancelled immediately', [
                'organization_id' => $subscription->organization_id,
                'reason' => $reason,
            ]);
        } else {
            $subscription->update([
                'cancel_at_period_end' => true,
                'cancellation_reason' => $reason,
            ]);

            Log::info('Subscription scheduled for cancellation', [
                'organization_id' => $subscription->organization_id,
                'cancellation_date' => $subscription->current_period_end,
                'reason' => $reason,
            ]);
        }

        return $subscription->fresh();
    }

    /**
     * Reactivate cancelled subscription
     *
     * @param OrganizationSubscription $subscription
     * @return OrganizationSubscription
     */
    public function reactivateSubscription(OrganizationSubscription $subscription): OrganizationSubscription
    {
        if ($subscription->status === 'active') {
            throw new Exception('Subscription is already active');
        }

        $now = now();
        
        $subscription->update([
            'status' => 'active',
            'activated_at' => $now,
            'cancelled_at' => null,
            'cancel_at_period_end' => false,
            'cancellation_reason' => null,
            'suspended_at' => null,
            'suspension_reason' => null,
            'current_period_start' => $now,
            'current_period_end' => $this->calculatePeriodEnd($now, $subscription->billing_cycle),
            'next_billing_date' => $this->calculatePeriodEnd($now, $subscription->billing_cycle),
        ]);

        $subscription->organization->update([
            'subscription_status' => 'active',
            'subscription_ends_at' => $subscription->current_period_end,
        ]);

        Log::info('Subscription reactivated', [
            'organization_id' => $subscription->organization_id,
        ]);

        return $subscription->fresh();
    }

    /**
     * Suspend subscription
     *
     * @param OrganizationSubscription $subscription
     * @param string|null $reason
     * @return OrganizationSubscription
     */
    public function suspendSubscription(OrganizationSubscription $subscription, ?string $reason = null): OrganizationSubscription
    {
        $subscription->update([
            'status' => 'suspended',
            'suspended_at' => now(),
            'suspension_reason' => $reason,
        ]);

        $subscription->organization->update([
            'subscription_status' => 'suspended',
        ]);

        Log::warning('Subscription suspended', [
            'organization_id' => $subscription->organization_id,
            'reason' => $reason,
        ]);

        return $subscription->fresh();
    }

    /**
     * Renew subscription for next billing period
     *
     * @param OrganizationSubscription $subscription
     * @return OrganizationSubscription
     */
    public function renewSubscription(OrganizationSubscription $subscription): OrganizationSubscription
    {
        $newPeriodStart = $subscription->current_period_end;
        $newPeriodEnd = $this->calculatePeriodEnd($newPeriodStart, $subscription->billing_cycle);

        $subscription->update([
            'current_period_start' => $newPeriodStart,
            'current_period_end' => $newPeriodEnd,
            'next_billing_date' => $newPeriodEnd,
        ]);

        $subscription->organization->update([
            'subscription_ends_at' => $newPeriodEnd,
        ]);

        Log::info('Subscription renewed', [
            'organization_id' => $subscription->organization_id,
            'new_period_end' => $newPeriodEnd,
        ]);

        return $subscription->fresh();
    }

    /**
     * Check and process expired trials
     *
     * @return int Number of trials processed
     */
    public function processExpiredTrials(): int
    {
        $expiredTrials = OrganizationSubscription::expiredTrials()->get();
        $count = 0;

        foreach ($expiredTrials as $subscription) {
            try {
                // Check if there's a paid invoice
                $paidInvoice = Invoice::where('organization_id', $subscription->organization_id)
                    ->where('status', 'paid')
                    ->where('type', 'subscription')
                    ->exists();

                if ($paidInvoice) {
                    // Convert to paid
                    $this->convertTrialToPaid($subscription);
                } else {
                    // Expire the subscription
                    $subscription->update(['status' => 'expired']);
                    $subscription->organization->update(['subscription_status' => 'expired']);
                    
                    Log::info('Trial expired', [
                        'organization_id' => $subscription->organization_id,
                    ]);
                }

                $count++;
            } catch (Exception $e) {
                Log::error('Failed to process expired trial', [
                    'subscription_id' => $subscription->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return $count;
    }

    /**
     * Calculate period end date based on billing cycle
     *
     * @param Carbon $startDate
     * @param string $billingCycle
     * @return Carbon
     */
    protected function calculatePeriodEnd(Carbon $startDate, string $billingCycle): Carbon
    {
        return match($billingCycle) {
            'monthly' => $startDate->copy()->addMonth(),
            'annual' => $startDate->copy()->addYear(),
            default => $startDate->copy()->addMonth(),
        };
    }

    /**
     * Get subscription summary for organization
     *
     * @param Organization $organization
     * @return array
     */
    public function getSubscriptionSummary(Organization $organization): array
    {
        $subscription = $organization->subscription;
        
        if (!$subscription) {
            return [
                'has_subscription' => false,
                'status' => 'none',
            ];
        }

        $plan = $subscription->plan;

        return [
            'has_subscription' => true,
            'status' => $subscription->status,
            'plan' => [
                'name' => $plan->name,
                'price' => $subscription->billing_cycle === 'annual' ? $plan->price_annual : $plan->price_monthly,
                'billing_cycle' => $subscription->billing_cycle,
            ],
            'trial' => [
                'on_trial' => $subscription->onTrial(),
                'days_remaining' => $subscription->trialDaysRemaining(),
                'ends_at' => $subscription->trial_ends_at?->format('Y-m-d'),
            ],
            'billing' => [
                'current_period_start' => $subscription->current_period_start?->format('Y-m-d'),
                'current_period_end' => $subscription->current_period_end?->format('Y-m-d'),
                'next_billing_date' => $subscription->next_billing_date?->format('Y-m-d'),
            ],
            'cancellation' => [
                'cancel_at_period_end' => $subscription->cancel_at_period_end,
                'cancelled_at' => $subscription->cancelled_at?->format('Y-m-d'),
                'reason' => $subscription->cancellation_reason,
            ],
        ];
    }
}
