<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;

class SecurityMonitoringService
{
    /**
     * Log security event
     */
    public function logSecurityEvent(string $event, array $data = [], string $level = 'warning'): void
    {
        $logData = array_merge([
            'event' => $event,
            'timestamp' => now()->toISOString(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'user_id' => auth()->id(),
        ], $data);

        Log::channel('security')->{$level}($event, $logData);
    }

    /**
     * Monitor failed login attempts
     */
    public function monitorFailedLogin(string $email, string $ip): void
    {
        $key = "failed_login_{$ip}_{$email}";
        $attempts = Cache::get($key, 0) + 1;
        
        Cache::put($key, $attempts, now()->addHours(1));

        $this->logSecurityEvent('failed_login_attempt', [
            'email' => $email,
            'attempts' => $attempts,
            'ip_address' => $ip,
        ]);

        // Alert if too many attempts
        if ($attempts >= 5) {
            $this->alertSuspiciousActivity('Multiple failed login attempts', [
                'email' => $email,
                'ip_address' => $ip,
                'attempts' => $attempts,
            ]);
        }
    }

    /**
     * Monitor suspicious input patterns
     */
    public function monitorSuspiciousInput(Request $request): void
    {
        $inputSanitizer = new InputSanitizationService();
        $suspiciousInputs = [];

        foreach ($request->all() as $key => $value) {
            if (is_string($value)) {
                $safety = $inputSanitizer->isInputSafe($value);
                
                if (!$safety['safe']) {
                    $suspiciousInputs[$key] = [
                        'value' => $value,
                        'sql_injection' => $safety['sql_injection'],
                        'xss' => $safety['xss'],
                    ];
                }
            }
        }

        if (!empty($suspiciousInputs)) {
            $this->logSecurityEvent('suspicious_input_detected', [
                'url' => $request->fullUrl(),
                'method' => $request->method(),
                'suspicious_inputs' => $suspiciousInputs,
            ], 'error');

            $this->alertSuspiciousActivity('Suspicious input patterns detected', [
                'url' => $request->fullUrl(),
                'inputs' => array_keys($suspiciousInputs),
            ]);
        }
    }

    /**
     * Monitor unusual access patterns
     */
    public function monitorAccessPattern(User $user, Request $request): void
    {
        $ip = $request->ip();
        $userAgent = $request->userAgent();
        
        // Check for new IP address
        $lastIp = Cache::get("user_last_ip_{$user->id}");
        if ($lastIp && $lastIp !== $ip) {
            $this->logSecurityEvent('new_ip_access', [
                'user_id' => $user->id,
                'previous_ip' => $lastIp,
                'new_ip' => $ip,
            ]);
        }
        
        // Check for new user agent
        $lastUserAgent = Cache::get("user_last_agent_{$user->id}");
        if ($lastUserAgent && $lastUserAgent !== $userAgent) {
            $this->logSecurityEvent('new_device_access', [
                'user_id' => $user->id,
                'previous_agent' => $lastUserAgent,
                'new_agent' => $userAgent,
            ]);
        }

        // Update cache
        Cache::put("user_last_ip_{$user->id}", $ip, now()->addDays(30));
        Cache::put("user_last_agent_{$user->id}", $userAgent, now()->addDays(30));
    }

    /**
     * Monitor rate limiting violations
     */
    public function monitorRateLimit(string $key, int $maxAttempts, int $decayMinutes): bool
    {
        $attempts = Cache::get($key, 0);
        
        if ($attempts >= $maxAttempts) {
            $this->logSecurityEvent('rate_limit_exceeded', [
                'key' => $key,
                'attempts' => $attempts,
                'max_attempts' => $maxAttempts,
            ]);
            
            return false;
        }

        Cache::put($key, $attempts + 1, now()->addMinutes($decayMinutes));
        return true;
    }

    /**
     * Monitor file upload security
     */
    public function monitorFileUpload($file, User $user): array
    {
        $issues = [];
        
        // Check file size
        $maxSize = 10 * 1024 * 1024; // 10MB
        if ($file->getSize() > $maxSize) {
            $issues[] = 'file_too_large';
        }

        // Check file extension
        $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx'];
        $extension = strtolower($file->getClientOriginalExtension());
        
        if (!in_array($extension, $allowedExtensions)) {
            $issues[] = 'invalid_file_type';
        }

        // Check MIME type
        $allowedMimes = [
            'image/jpeg', 'image/png', 'image/gif',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
        ];
        
        if (!in_array($file->getMimeType(), $allowedMimes)) {
            $issues[] = 'invalid_mime_type';
        }

        // Log if issues found
        if (!empty($issues)) {
            $this->logSecurityEvent('suspicious_file_upload', [
                'user_id' => $user->id,
                'filename' => $file->getClientOriginalName(),
                'size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'extension' => $extension,
                'issues' => $issues,
            ]);
        }

        return [
            'safe' => empty($issues),
            'issues' => $issues,
        ];
    }

    /**
     * Alert administrators about suspicious activity
     */
    private function alertSuspiciousActivity(string $title, array $data): void
    {
        // In a real application, you might send emails, Slack notifications, etc.
        $this->logSecurityEvent('security_alert', [
            'title' => $title,
            'data' => $data,
        ], 'critical');

        // Store alert for dashboard
        Cache::put(
            'security_alert_' . time(),
            [
                'title' => $title,
                'data' => $data,
                'timestamp' => now(),
            ],
            now()->addDays(7)
        );
    }

    /**
     * Get recent security alerts
     */
    public function getRecentAlerts(int $limit = 10): array
    {
        $alerts = [];
        $keys = Cache::getRedis()->keys('security_alert_*');
        
        foreach ($keys as $key) {
            $alert = Cache::get($key);
            if ($alert) {
                $alerts[] = $alert;
            }
        }

        // Sort by timestamp and limit
        usort($alerts, function ($a, $b) {
            return $b['timestamp'] <=> $a['timestamp'];
        });

        return array_slice($alerts, 0, $limit);
    }

    /**
     * Check if IP is whitelisted
     */
    public function isIpWhitelisted(string $ip): bool
    {
        $whitelist = config('security.ip_whitelist', []);
        
        foreach ($whitelist as $whitelistedIp) {
            if ($this->ipMatches($ip, $whitelistedIp)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if IP matches pattern (supports CIDR notation)
     */
    private function ipMatches(string $ip, string $pattern): bool
    {
        if ($ip === $pattern) {
            return true;
        }

        // Check CIDR notation
        if (strpos($pattern, '/') !== false) {
            list($subnet, $mask) = explode('/', $pattern);
            
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
                return $this->ipv4InRange($ip, $subnet, $mask);
            }
        }

        return false;
    }

    /**
     * Check if IPv4 address is in range
     */
    private function ipv4InRange(string $ip, string $subnet, int $mask): bool
    {
        $ipLong = ip2long($ip);
        $subnetLong = ip2long($subnet);
        $maskLong = -1 << (32 - $mask);
        
        return ($ipLong & $maskLong) === ($subnetLong & $maskLong);
    }

    /**
     * Generate security report
     */
    public function generateSecurityReport(int $days = 7): array
    {
        // This would typically query a dedicated security events table
        // For now, we'll return a basic structure
        
        return [
            'period' => $days,
            'failed_logins' => $this->getFailedLoginCount($days),
            'suspicious_inputs' => $this->getSuspiciousInputCount($days),
            'new_device_logins' => $this->getNewDeviceLoginCount($days),
            'rate_limit_violations' => $this->getRateLimitViolationCount($days),
            'recent_alerts' => $this->getRecentAlerts(),
        ];
    }

    /**
     * Get failed login count for period
     */
    private function getFailedLoginCount(int $days): int
    {
        // This would query actual log data
        return rand(0, 50); // Placeholder
    }

    /**
     * Get suspicious input count for period
     */
    private function getSuspiciousInputCount(int $days): int
    {
        // This would query actual log data
        return rand(0, 10); // Placeholder
    }

    /**
     * Get new device login count for period
     */
    private function getNewDeviceLoginCount(int $days): int
    {
        // This would query actual log data
        return rand(0, 20); // Placeholder
    }

    /**
     * Get rate limit violation count for period
     */
    private function getRateLimitViolationCount(int $days): int
    {
        // This would query actual log data
        return rand(0, 5); // Placeholder
    }
}