<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\Payroll;
use App\Models\Contract;
use App\Models\Attendance;
use App\Models\EmployeeLoan;
use App\Models\EmployeeAdvance;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class PayrollService
{
    /**
     * Generate payroll for all employees for a given month
     */
    public function generateMonthlyPayroll(int $year, int $month, array $employeeIds = []): Collection
    {
        $query = Employee::where('status', 'active')
            ->whereHas('activeContract');
            
        if (!empty($employeeIds)) {
            $query->whereIn('id', $employeeIds);
        }
        
        $employees = $query->get();
        $payrolls = collect();
        
        foreach ($employees as $employee) {
            try {
                $payroll = $this->generateEmployeePayroll($employee, $year, $month);
                $payrolls->push($payroll);
            } catch (\Exception $e) {
                // Log error and continue with next employee
                \Log::error("Failed to generate payroll for employee {$employee->id}: " . $e->getMessage());
            }
        }
        
        return $payrolls;
    }
    
    /**
     * Generate payroll for a specific employee
     */
    public function generateEmployeePayroll(Employee $employee, int $year, int $month): Payroll
    {
        // Check if payroll already exists
        $existingPayroll = Payroll::where('employee_id', $employee->id)
            ->where('year', $year)
            ->where('month', $month)
            ->first();
            
        if ($existingPayroll) {
            throw new \Exception("Payroll already exists for {$employee->full_name} for {$year}-{$month}");
        }
        
        $contract = $employee->activeContract;
        if (!$contract) {
            throw new \Exception("Employee {$employee->full_name} does not have an active contract");
        }
        
        // Create payroll record
        $payroll = new Payroll([
            'employee_id' => $employee->id,
            'employment_type' => $contract->employment_type,
            'contract_id' => $contract->id,
            'month' => $month,
            'year' => $year,
            'payroll_number' => Payroll::generatePayrollNumber($employee, $year, $month),
            'organization_id' => $employee->organization_id,
        ]);
        
        // Load attendance data and calculate hours/days worked
        $this->loadAttendanceData($payroll, $employee, $year, $month);
        
        // Load employee allowances and deductions
        $this->loadEmployeeAllowances($payroll, $employee);
        $this->loadEmployeeDeductions($payroll, $employee);
        
        // Calculate payroll based on employment type
        $this->calculatePayrollByEmploymentType($payroll, $contract);
        
        $payroll->save();
        
        // Record loan repayments
        $this->recordLoanRepayments($payroll, $employee);
        
        // Initialize approval workflow
        app(PayrollApprovalService::class)->initializeApprovalWorkflow($payroll);
        
        return $payroll;
    }    
 
   /**
     * Load attendance data for payroll calculation
     */
    protected function loadAttendanceData(Payroll $payroll, Employee $employee, int $year, int $month): void
    {
        $attendances = $employee->attendances()
            ->byMonth($year, $month)
            ->approved()
            ->notInPayroll()
            ->get();
            
        $totalHours = 0;
        $totalRegularHours = 0;
        $totalOvertimeHours = 0;
        $totalDaysWorked = 0;
        
        foreach ($attendances as $attendance) {
            // Ensure attendance hours are calculated
            if (!$attendance->regular_hours) {
                $attendance->calculateDetailedHours();
            }
            
            $totalHours += $attendance->worked_hours;
            $totalRegularHours += $attendance->regular_hours;
            $totalOvertimeHours += $attendance->overtime_hours;
            $totalDaysWorked++;
        }
        
        $payroll->hours_worked = $totalHours;
        $payroll->regular_hours = $totalRegularHours;
        $payroll->overtime_hours = $totalOvertimeHours;
        $payroll->days_worked = $totalDaysWorked;
        $payroll->present_days = $totalDaysWorked;
        
        // Calculate expected working days
        $payroll->expected_working_days = $this->calculateExpectedWorkingDays($year, $month);
        $payroll->working_days = $payroll->expected_working_days; // Set working days to expected days
        $payroll->absent_days = max(0, $payroll->expected_working_days - $payroll->present_days);
        $payroll->leave_days = 0; // Initialize leave days to 0
        
        // Mark attendances as included in payroll
        foreach ($attendances as $attendance) {
            $attendance->markAsIncludedInPayroll($payroll);
        }
    }
    
    /**
     * Load employee allowances
     */
    protected function loadEmployeeAllowances(Payroll $payroll, Employee $employee): void
    {
        $contract = $employee->activeContract;
        
        // Base allowances from employee record
        $payroll->house_allowance = $employee->house_allowance ?? 0;
        $payroll->transport_allowance = $employee->transport_allowance ?? 0;
        $payroll->medical_allowance = $employee->medical_allowance ?? 0;
        
        // Check contract eligibility for allowances
        if ($contract) {
            $eligibleAllowances = $contract->eligible_allowances ?? [];
            
            if (!empty($eligibleAllowances)) {
                if (!in_array('house', $eligibleAllowances)) {
                    $payroll->house_allowance = 0;
                }
                if (!in_array('transport', $eligibleAllowances)) {
                    $payroll->transport_allowance = 0;
                }
                if (!in_array('medical', $eligibleAllowances)) {
                    $payroll->medical_allowance = 0;
                }
            }
        }
        
        // Calculate total allowances
        $payroll->allowances = $payroll->house_allowance + 
                              $payroll->transport_allowance + 
                              $payroll->medical_allowance + 
                              $payroll->other_allowances;
    }
    
    /**
     * Load employee deductions
     */
    protected function loadEmployeeDeductions(Payroll $payroll, Employee $employee): void
    {
        // Load from employee record or calculate based on loans, advances, etc.
        $payroll->loan_deductions = $this->calculateLoanDeductions($employee, $payroll->year, $payroll->month);
        $payroll->advance_deductions = $this->calculateAdvanceDeductions($employee, $payroll->year, $payroll->month);
        $payroll->disciplinary_deductions = $this->calculateDisciplinaryDeductions($employee, $payroll->year, $payroll->month);
        $payroll->helb_deduction = $employee->helb_deduction ?? 0;
    }    

    /**
     * Calculate payroll based on employment type
     */
    protected function calculatePayrollByEmploymentType(Payroll $payroll, Contract $contract): void
    {
        // Set rates from contract
        $payroll->hourly_rate = $contract->effective_hourly_rate;
        $payroll->daily_rate = $contract->effective_daily_rate;
        $payroll->overtime_rate = $contract->overtime_rate;
        
        // Calculate basic pay based on employment type
        switch ($contract->employment_type) {
            case 'permanent':
                $this->calculatePermanentPayroll($payroll, $contract);
                break;
                
            case 'contract':
            case 'freelance':
                $this->calculateContractPayroll($payroll, $contract);
                break;
                
            case 'casual':
                $this->calculateCasualPayroll($payroll, $contract);
                break;
                
            case 'intern':
            case 'part_time':
                $this->calculateHourlyPayroll($payroll, $contract);
                break;
        }
        
        // Use Kenya Tax Calculator with employment type support
        $salaryComponents = [
            'basic_salary' => $payroll->contract_basic_pay,
            'house_allowance' => $payroll->house_allowance,
            'transport_allowance' => $payroll->transport_allowance,
            'medical_allowance' => $payroll->medical_allowance,
            'other_allowances' => $payroll->other_allowances,
            'bonuses' => $payroll->bonuses,
            'performance_bonus' => $payroll->performance_bonus,
            'merit_increase' => $payroll->merit_increase,
            'commission' => $payroll->commission,
            'overtime_pay' => $payroll->overtime_pay,
            'helb_deduction' => $payroll->helb_deduction,
            'loan_deductions' => $payroll->loan_deductions,
            'advance_deductions' => $payroll->advance_deductions,
            'disciplinary_deductions' => $payroll->disciplinary_deductions,
            'other_deductions' => $payroll->other_deductions,
            'statutory_deductions_applicable' => $contract->statutory_deductions_applicable,
            'withholding_tax_rate' => $contract->withholding_tax_rate,
        ];
        
        $calculations = KenyaTaxCalculator::calculatePayrollByEmploymentType(
            $salaryComponents, 
            $contract->employment_type
        );
        
        // Update payroll with calculated values
        foreach ($calculations as $field => $value) {
            if (in_array($field, $payroll->getFillable())) {
                $payroll->$field = $value;
            }
        }
        
        // Store calculation details
        $payroll->calculation_breakdown = [
            'employment_type' => $contract->employment_type,
            'calculation_method' => $this->getCalculationMethod($contract),
            'rates_used' => [
                'hourly_rate' => $contract->effective_hourly_rate,
                'daily_rate' => $contract->effective_daily_rate,
                'overtime_multiplier' => $contract->overtime_rate_multiplier,
            ],
            'hours_breakdown' => [
                'regular_hours' => $payroll->regular_hours,
                'overtime_hours' => $payroll->overtime_hours,
                'total_hours' => $payroll->hours_worked,
            ],
            'days_breakdown' => [
                'days_worked' => $payroll->days_worked,
                'expected_days' => $payroll->expected_working_days,
                'absent_days' => $payroll->absent_days,
            ],
            'calculated_at' => now()->toISOString(),
        ];
    }    
    
protected function calculatePermanentPayroll(Payroll $payroll, Contract $contract): void
    {
        $monthlySalary = $contract->salary;
        
        // Check if salary should be prorated
        if ($this->shouldProrateSalary($payroll, $contract)) {
            $prorationFactor = $payroll->present_days / $payroll->expected_working_days;
            $monthlySalary = $monthlySalary * $prorationFactor;
            $payroll->prorated = true;
            $payroll->proration_reason = "Prorated based on {$payroll->present_days}/{$payroll->expected_working_days} working days";
        }
        
        $payroll->basic_salary = $monthlySalary;
        $payroll->contract_basic_pay = $monthlySalary;
        
        // Calculate overtime if applicable
        if ($contract->overtime_eligible && $payroll->overtime_hours > 0) {
            $payroll->overtime_pay = $payroll->overtime_hours * $contract->overtime_rate;
        }
    }
    
    protected function calculateContractPayroll(Payroll $payroll, Contract $contract): void
    {
        if ($contract->payment_frequency === 'hourly') {
            // Hourly-based calculation
            $payroll->hourly_pay = $payroll->regular_hours * $contract->effective_hourly_rate;
            $payroll->overtime_pay = $payroll->overtime_hours * $contract->overtime_rate;
            $payroll->contract_basic_pay = $payroll->hourly_pay + $payroll->overtime_pay;
        } elseif ($contract->payment_frequency === 'daily') {
            // Daily-based calculation
            $payroll->daily_pay = $payroll->days_worked * $contract->effective_daily_rate;
            $payroll->contract_basic_pay = $payroll->daily_pay;
        } else {
            // Monthly contract
            $payroll->contract_basic_pay = $contract->salary;
        }
        
        $payroll->basic_salary = $payroll->contract_basic_pay;
    }
    
    protected function calculateCasualPayroll(Payroll $payroll, Contract $contract): void
    {
        $payroll->daily_pay = $payroll->days_worked * $contract->effective_daily_rate;
        $payroll->contract_basic_pay = $payroll->daily_pay;
        $payroll->basic_salary = $payroll->contract_basic_pay;
    }
    
    protected function calculateHourlyPayroll(Payroll $payroll, Contract $contract): void
    {
        $payroll->hourly_pay = $payroll->regular_hours * $contract->effective_hourly_rate;
        
        if ($contract->overtime_eligible && $payroll->overtime_hours > 0) {
            $payroll->overtime_pay = $payroll->overtime_hours * $contract->overtime_rate;
        }
        
        $payroll->contract_basic_pay = $payroll->hourly_pay + $payroll->overtime_pay;
        $payroll->basic_salary = $payroll->contract_basic_pay;
    }
    
    protected function shouldProrateSalary(Payroll $payroll, Contract $contract): bool
    {
        // Prorate if employee worked less than 80% of expected days
        $attendanceRate = $payroll->present_days / $payroll->expected_working_days;
        return $attendanceRate < 0.8;
    }
    
    protected function getCalculationMethod(Contract $contract): string
    {
        return match($contract->employment_type) {
            'permanent' => 'monthly_salary_with_proration',
            'contract', 'freelance' => $contract->payment_frequency . '_rate',
            'casual' => 'daily_rate',
            'intern', 'part_time' => 'hourly_rate',
            default => 'monthly_salary'
        };
    }    

    protected function calculateExpectedWorkingDays(int $year, int $month): int
    {
        $startDate = Carbon::create($year, $month, 1);
        $endDate = $startDate->copy()->endOfMonth();
        
        $workingDays = 0;
        $current = $startDate->copy();
        
        while ($current <= $endDate) {
            // Skip weekends
            if (!in_array($current->dayOfWeek, [0, 6])) {
                $workingDays++;
            }
            $current->addDay();
        }
        
        return $workingDays;
    }
    
    protected function calculateLoanDeductions(Employee $employee, int $year, int $month): float
    {
        $totalDeduction = 0;
        
        // Get active loans
        $activeLoans = EmployeeLoan::where('employee_id', $employee->id)
            ->where('status', 'active')
            ->where('first_deduction_date', '<=', "{$year}-{$month}-01")
            ->get();
        
        foreach ($activeLoans as $loan) {
            if ($loan->balance > 0) {
                $deduction = min($loan->monthly_deduction, $loan->balance);
                $totalDeduction += $deduction;
            }
        }
        
        return $totalDeduction;
    }
    
    protected function calculateAdvanceDeductions(Employee $employee, int $year, int $month): float
    {
        $totalDeduction = 0;
        
        // Get active advances
        $activeAdvances = EmployeeAdvance::where('employee_id', $employee->id)
            ->where('status', 'active')
            ->where('first_deduction_date', '<=', "{$year}-{$month}-01")
            ->get();
        
        foreach ($activeAdvances as $advance) {
            if ($advance->balance > 0) {
                $deduction = min($advance->monthly_deduction, $advance->balance);
                $totalDeduction += $deduction;
            }
        }
        
        return $totalDeduction;
    }
    
    protected function calculateDisciplinaryDeductions(Employee $employee, int $year, int $month): float
    {
        // This would typically query a disciplinary actions table
        // For now, return from employee record
        return $employee->disciplinary_deductions ?? 0;
    }

    /**
     * Record loan repayments for this payroll
     */
    protected function recordLoanRepayments(Payroll $payroll, Employee $employee): void
    {
        // Get active loans that should be deducted
        $activeLoans = EmployeeLoan::where('employee_id', $employee->id)
            ->where('status', 'active')
            ->where('first_deduction_date', '<=', "{$payroll->year}-{$payroll->month}-01")
            ->get();
        
        foreach ($activeLoans as $loan) {
            if ($loan->balance > 0) {
                $deduction = min($loan->monthly_deduction, $loan->balance);
                $loan->recordRepayment($deduction, $payroll->id, 'payroll_deduction');
            }
        }

        // Get active advances that should be deducted
        $activeAdvances = EmployeeAdvance::where('employee_id', $employee->id)
            ->where('status', 'active')
            ->where('first_deduction_date', '<=', "{$payroll->year}-{$payroll->month}-01")
            ->get();
        
        foreach ($activeAdvances as $advance) {
            if ($advance->balance > 0) {
                $deduction = min($advance->monthly_deduction, $advance->balance);
                
                // Update advance
                $advance->amount_repaid += $deduction;
                $advance->balance -= $deduction;
                
                if ($advance->balance <= 0) {
                    $advance->status = 'completed';
                }
                
                $advance->save();
            }
        }
    }
    
    /**
     * Approve payroll for processing
     */
    public function approvePayroll(Payroll $payroll, $approvedBy): bool
    {
        $payroll->approval_status = 'approved';
        $payroll->approved_by = $approvedBy;
        $payroll->approved_at = now();
        
        return $payroll->save();
    }
    
    /**
     * Process approved payroll for payment
     */
    public function processPayroll(Payroll $payroll, $processedBy): bool
    {
        if ($payroll->approval_status !== 'approved') {
            throw new \Exception('Payroll must be approved before processing');
        }
        
        $payroll->status = 'processed';
        $payroll->processed_by = $processedBy;
        $payroll->processed_at = now();
        
        return $payroll->save();
    }
    
    /**
     * Generate payroll summary for a period
     */
    public function generatePayrollSummary(int $year, int $month, $organizationId = null): array
    {
        $query = Payroll::where('year', $year)->where('month', $month);
        
        if ($organizationId) {
            $query->where('organization_id', $organizationId);
        }
        
        $payrolls = $query->get();
        
        return [
            'period' => "{$year}-{$month}",
            'total_employees' => $payrolls->count(),
            'total_gross_salary' => $payrolls->sum('gross_salary'),
            'total_net_salary' => $payrolls->sum('net_salary'),
            'total_deductions' => $payrolls->sum('total_deductions'),
            'total_paye_tax' => $payrolls->sum('paye_tax'),
            'total_nhif' => $payrolls->sum('nhif_deduction'),
            'total_nssf_employee' => $payrolls->sum('nssf_employee'),
            'total_housing_levy' => $payrolls->sum('housing_levy'),
            'by_employment_type' => $payrolls->groupBy('employment_type')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'total_gross' => $group->sum('gross_salary'),
                    'total_net' => $group->sum('net_salary'),
                ];
            }),
            'by_status' => $payrolls->groupBy('status')->map->count(),
        ];
    }
}