<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Spatie\OneTimePasswords\Models\OneTimePassword;

class OtpService
{
    /**
     * Generate and send OTP to user
     *
     * @param User $user
     * @return bool
     */
    public function generateAndSendOtp(User $user): bool
    {
        // Check if can resend (rate limiting)
        if (!$this->canResend($user)) {
            return false;
        }

        try {
            // Generate OTP using Spatie package (10 minutes expiry)
            $otp = OneTimePassword::generateFor($user, expiresInMinutes: 10);

            // Update user with last sent timestamp
            $user->update([
                'last_otp_sent_at' => now(),
            ]);

            // Send OTP via email (Spatie uses 'password' field for the code)
            Mail::to($user->email)->send(new \App\Mail\VerifyEmailOtp($user, $otp->password));
            
            Log::info('OTP sent to user', [
                'user_id' => $user->id,
                'email' => $user->email,
                'otp_id' => $otp->id,
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to send OTP email', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Validate OTP code
     *
     * @param User $user
     * @param string $code
     * @return bool
     */
    public function validateOtp(User $user, string $code): bool
    {
        try {
            // Get the latest OTP for this user (Spatie uses 'password' field)
            $otp = OneTimePassword::where('authenticatable_type', User::class)
                ->where('authenticatable_id', $user->id)
                ->where('password', $code)
                ->where('expires_at', '>', now())
                ->first();

            if (!$otp) {
                Log::warning('Invalid or expired OTP attempt', [
                    'user_id' => $user->id,
                ]);
                return false;
            }

            // Check if OTP is expired
            if ($otp->isExpired()) {
                Log::warning('Expired OTP attempt', [
                    'user_id' => $user->id,
                    'otp_id' => $otp->id,
                ]);
                return false;
            }

            // Mark email as verified
            $user->update([
                'email_verified_at' => now(),
                'must_verify_email' => false,
            ]);

            // Delete the OTP after successful validation
            $otp->delete();

            Log::info('OTP validated successfully', [
                'user_id' => $user->id,
                'otp_id' => $otp->id,
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('OTP validation error', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
            ]);

            return false;
        }
    }

    /**
     * Check if OTP can be resent (rate limiting)
     *
     * @param User $user
     * @return bool
     */
    public function canResend(User $user): bool
    {
        // No previous OTP sent
        if (!$user->last_otp_sent_at) {
            return true;
        }

        // Check if 1 minute has passed since last OTP
        return $user->last_otp_sent_at->diffInSeconds(now()) >= 60;
    }

    /**
     * Get seconds until can resend
     *
     * @param User $user
     * @return int
     */
    public function secondsUntilCanResend(User $user): int
    {
        if (!$user->last_otp_sent_at) {
            return 0;
        }

        $elapsed = $user->last_otp_sent_at->diffInSeconds(now());
        return max(0, 60 - $elapsed);
    }

    /**
     * Get remaining attempts for user
     *
     * @param User $user
     * @return int
     */
    public function getRemainingAttempts(User $user): int
    {
        try {
            // Get the latest OTP for this user
            $otp = OneTimePassword::where('authenticatable_type', User::class)
                ->where('authenticatable_id', $user->id)
                ->where('expires_at', '>', now())
                ->latest()
                ->first();

            // If no active OTP, return max attempts
            if (!$otp) {
                return 5;
            }

            // Count failed attempts (we'll track this differently)
            // For now, return 5 as Spatie doesn't track attempts by default
            return 5;
        } catch (\Exception $e) {
            return 5;
        }
    }
}
