<?php

namespace App\Services;

use App\Models\Organization;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Mail;

class OrganizationMailService
{
    /**
     * Configure mail settings for a specific organization
     */
    public static function configureForOrganization(Organization $organization): void
    {
        if (!$organization->use_custom_mail) {
            return;
        }

        // Set organization-specific mail configuration
        Config::set('mail.mailers.smtp', [
            'transport' => 'smtp',
            'host' => $organization->mail_host,
            'port' => $organization->mail_port,
            'encryption' => $organization->mail_encryption,
            'username' => $organization->mail_username,
            'password' => $organization->mail_password,
            'timeout' => null,
        ]);

        Config::set('mail.from', [
            'address' => $organization->mail_from_address ?? config('mail.from.address'),
            'name' => $organization->mail_from_name ?? $organization->name,
        ]);

        // Purge the mail manager to force reconfiguration
        Mail::purge('smtp');
    }

    /**
     * Test organization email configuration
     */
    public static function testConfiguration(Organization $organization, string $testEmail): bool
    {
        try {
            self::configureForOrganization($organization);
            
            Mail::raw('This is a test email from ' . $organization->name, function ($message) use ($testEmail, $organization) {
                $message->to($testEmail)
                    ->subject('Test Email - ' . $organization->name);
            });

            return true;
        } catch (\Exception $e) {
            \Log::error('Email test failed for organization ' . $organization->id, [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }
}
