<?php

namespace App\Services;

use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Organization;
use Iankumu\Mpesa\Facades\Mpesa;
use Illuminate\Support\Facades\Log;
use Exception;

class MpesaService
{
    /**
     * Initiate STK Push for subscription payment
     *
     * @param Invoice $invoice
     * @param string $phoneNumber Format: 254712345678
     * @return array
     */
    public function initiateSubscriptionPayment(Invoice $invoice, string $phoneNumber): array
    {
        try {
            // Format phone number (remove + and spaces)
            $phone = $this->formatPhoneNumber($phoneNumber);
            
            // Validate phone number
            if (!$this->isValidKenyanPhone($phone)) {
                return [
                    'success' => false,
                    'message' => 'Invalid phone number format. Use format: 254712345678',
                ];
            }

            // Create payment record
            $payment = Payment::create([
                'invoice_id' => $invoice->id,
                'organization_id' => $invoice->organization_id,
                'amount' => $invoice->total,
                'currency' => 'KES',
                'payment_method' => 'mpesa',
                'status' => 'pending',
                'mpesa_phone_number' => $phone,
                'initiated_at' => now(),
                'ip_address' => request()->ip(),
            ]);

            // Initiate STK Push
            $response = Mpesa::stkpush(
                phone: $phone,
                amount: (int) $invoice->total, // M-Pesa requires integer
                reference: $invoice->invoice_number,
                description: "Payment for {$invoice->invoice_number}"
            );

            // Log the response
            Log::info('M-Pesa STK Push initiated', [
                'invoice_id' => $invoice->id,
                'payment_id' => $payment->id,
                'phone' => $phone,
                'amount' => $invoice->total,
                'response' => $response,
            ]);

            // Update payment with M-Pesa details
            if (isset($response['CheckoutRequestID'])) {
                $payment->update([
                    'mpesa_checkout_request_id' => $response['CheckoutRequestID'],
                    'mpesa_merchant_request_id' => $response['MerchantRequestID'] ?? null,
                    'gateway_response' => $response,
                    'status' => 'processing',
                ]);

                return [
                    'success' => true,
                    'message' => 'Payment request sent. Please check your phone to complete payment.',
                    'payment_id' => $payment->id,
                    'checkout_request_id' => $response['CheckoutRequestID'],
                ];
            }

            // Handle error response
            $payment->markAsFailed('STK Push failed: ' . ($response['errorMessage'] ?? 'Unknown error'));

            return [
                'success' => false,
                'message' => $response['errorMessage'] ?? 'Failed to initiate payment. Please try again.',
            ];

        } catch (Exception $e) {
            Log::error('M-Pesa STK Push error', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            if (isset($payment)) {
                $payment->markAsFailed('Exception: ' . $e->getMessage());
            }

            return [
                'success' => false,
                'message' => 'An error occurred while processing payment. Please try again.',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ];
        }
    }

    /**
     * Process M-Pesa callback
     *
     * @param array $callbackData
     * @return bool
     */
    public function processCallback(array $callbackData): bool
    {
        try {
            Log::info('M-Pesa callback received', ['data' => $callbackData]);

            // Extract callback data
            $body = $callbackData['Body']['stkCallback'] ?? null;
            
            if (!$body) {
                Log::error('Invalid M-Pesa callback structure', ['data' => $callbackData]);
                return false;
            }

            $checkoutRequestId = $body['CheckoutRequestID'] ?? null;
            $resultCode = $body['ResultCode'] ?? null;
            $resultDesc = $body['ResultDesc'] ?? null;

            if (!$checkoutRequestId) {
                Log::error('Missing CheckoutRequestID in callback');
                return false;
            }

            // Find payment by checkout request ID
            $payment = Payment::where('mpesa_checkout_request_id', $checkoutRequestId)->first();

            if (!$payment) {
                Log::error('Payment not found for CheckoutRequestID', ['checkout_request_id' => $checkoutRequestId]);
                return false;
            }

            // Check result code
            if ($resultCode == 0) {
                // Payment successful
                $callbackMetadata = $body['CallbackMetadata']['Item'] ?? [];
                $mpesaData = $this->parseCallbackMetadata($callbackMetadata);

                $payment->markAsCompleted([
                    'mpesa_receipt_number' => $mpesaData['mpesa_receipt_number'] ?? null,
                    'mpesa_amount' => $mpesaData['amount'] ?? $payment->amount,
                    'mpesa_phone_number' => $mpesaData['phone_number'] ?? $payment->mpesa_phone_number,
                    'mpesa_result_code' => $resultCode,
                    'mpesa_result_desc' => $resultDesc,
                    'transaction_id' => $mpesaData['mpesa_receipt_number'] ?? null,
                    'gateway_response' => $callbackData,
                ]);

                Log::info('Payment completed successfully', [
                    'payment_id' => $payment->id,
                    'invoice_id' => $payment->invoice_id,
                    'receipt' => $mpesaData['mpesa_receipt_number'] ?? null,
                ]);

                // Activate subscription if this was a subscription payment
                $this->activateSubscriptionIfNeeded($payment);

                return true;
            } else {
                // Payment failed
                $payment->markAsFailed("M-Pesa Error: {$resultDesc} (Code: {$resultCode})");

                Log::warning('Payment failed', [
                    'payment_id' => $payment->id,
                    'result_code' => $resultCode,
                    'result_desc' => $resultDesc,
                ]);

                return false;
            }

        } catch (Exception $e) {
            Log::error('Error processing M-Pesa callback', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return false;
        }
    }

    /**
     * Query transaction status
     *
     * @param string $checkoutRequestId
     * @return array
     */
    public function queryTransactionStatus(string $checkoutRequestId): array
    {
        try {
            $response = Mpesa::stkquery($checkoutRequestId);

            Log::info('M-Pesa transaction status query', [
                'checkout_request_id' => $checkoutRequestId,
                'response' => $response,
            ]);

            return [
                'success' => true,
                'data' => $response,
            ];

        } catch (Exception $e) {
            Log::error('Error querying M-Pesa transaction status', [
                'checkout_request_id' => $checkoutRequestId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => 'Failed to query transaction status',
                'error' => config('app.debug') ? $e->getMessage() : null,
            ];
        }
    }

    /**
     * Format phone number to M-Pesa format (254XXXXXXXXX)
     *
     * @param string $phone
     * @return string
     */
    protected function formatPhoneNumber(string $phone): string
    {
        // Remove spaces, dashes, and plus sign
        $phone = preg_replace('/[\s\-\+]/', '', $phone);

        // If starts with 0, replace with 254
        if (substr($phone, 0, 1) === '0') {
            $phone = '254' . substr($phone, 1);
        }

        // If starts with 7 or 1, add 254
        if (strlen($phone) === 9 && in_array(substr($phone, 0, 1), ['7', '1'])) {
            $phone = '254' . $phone;
        }

        return $phone;
    }

    /**
     * Validate Kenyan phone number
     *
     * @param string $phone
     * @return bool
     */
    protected function isValidKenyanPhone(string $phone): bool
    {
        // Must be 12 digits starting with 254
        if (strlen($phone) !== 12 || substr($phone, 0, 3) !== '254') {
            return false;
        }

        // Next digit must be 7 or 1 (Safaricom/Airtel)
        $fourthDigit = substr($phone, 3, 1);
        if (!in_array($fourthDigit, ['7', '1'])) {
            return false;
        }

        return true;
    }

    /**
     * Parse callback metadata
     *
     * @param array $metadata
     * @return array
     */
    protected function parseCallbackMetadata(array $metadata): array
    {
        $data = [];

        foreach ($metadata as $item) {
            $name = $item['Name'] ?? null;
            $value = $item['Value'] ?? null;

            switch ($name) {
                case 'MpesaReceiptNumber':
                    $data['mpesa_receipt_number'] = $value;
                    break;
                case 'Amount':
                    $data['amount'] = $value;
                    break;
                case 'PhoneNumber':
                    $data['phone_number'] = $value;
                    break;
                case 'TransactionDate':
                    $data['transaction_date'] = $value;
                    break;
            }
        }

        return $data;
    }

    /**
     * Activate subscription if payment was for subscription
     *
     * @param Payment $payment
     * @return void
     */
    protected function activateSubscriptionIfNeeded(Payment $payment): void
    {
        try {
            $invoice = $payment->invoice;
            
            if (!$invoice || $invoice->type !== 'subscription') {
                return;
            }

            $organization = $invoice->organization;
            $subscription = $organization->subscription;

            if (!$subscription) {
                Log::warning('No subscription found for organization', [
                    'organization_id' => $organization->id,
                    'invoice_id' => $invoice->id,
                ]);
                return;
            }

            // If on trial, convert to active
            if ($subscription->status === 'trial') {
                $subscription->activate();
                
                // Update organization status
                $organization->update([
                    'subscription_status' => 'active',
                ]);

                Log::info('Subscription activated after payment', [
                    'organization_id' => $organization->id,
                    'subscription_id' => $subscription->id,
                ]);
            }

        } catch (Exception $e) {
            Log::error('Error activating subscription', [
                'payment_id' => $payment->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Get payment by checkout request ID
     *
     * @param string $checkoutRequestId
     * @return Payment|null
     */
    public function getPaymentByCheckoutRequestId(string $checkoutRequestId): ?Payment
    {
        return Payment::where('mpesa_checkout_request_id', $checkoutRequestId)->first();
    }
}
