<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Laravel\Facades\Image;

class LogoUploadService
{
    protected array $sizes = [
        'logo' => ['width' => 200, 'height' => null],
        'sidebar_logo' => ['width' => 44, 'height' => 44],
        'header_logo' => ['width' => null, 'height' => 28],
        'favicon' => ['width' => 32, 'height' => 32],
        'login_banner' => ['width' => 1200, 'height' => 400],
    ];

    /**
     * Upload and process logo
     */
    public function upload(UploadedFile $file, string $type, int $organizationId): string
    {
        $path = "logos/org_{$organizationId}/{$type}";
        
        // Delete old logo if exists
        Storage::disk('public')->deleteDirectory($path);
        
        // Generate filename
        $extension = $file->getClientOriginalExtension();
        $filename = $type . '_' . time() . '.' . $extension;
        $fullPath = "{$path}/{$filename}";
        
        // Get size constraints
        $size = $this->sizes[$type] ?? null;
        
        // For SVG files, just store them without processing
        if (strtolower($extension) === 'svg') {
            Storage::disk('public')->put($fullPath, file_get_contents($file));
            return Storage::disk('public')->url($fullPath);
        }
        
        // Process raster images
        try {
            $image = Image::read($file);
            
            if ($size) {
                if ($size['width'] && $size['height']) {
                    // Fit to exact dimensions
                    $image->cover($size['width'], $size['height']);
                } elseif ($size['height']) {
                    // Scale to height
                    $image->scale(height: $size['height']);
                } elseif ($size['width']) {
                    // Scale to width
                    $image->scale(width: $size['width']);
                }
            }
            
            // Save with optimization
            Storage::disk('public')->put($fullPath, (string) $image->encode());
            
        } catch (\Exception $e) {
            // If image processing fails, just store the original
            Storage::disk('public')->put($fullPath, file_get_contents($file));
        }
        
        return Storage::disk('public')->url($fullPath);
    }

    /**
     * Delete logo file
     */
    public function delete(?string $url): void
    {
        if ($url) {
            $path = str_replace('/storage/', '', parse_url($url, PHP_URL_PATH));
            Storage::disk('public')->delete($path);
        }
    }
}
