<?php

namespace App\Services;

use App\Models\Organization;
use App\Models\OrganizationSubscription;
use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Exception;

class BillingService
{
    /**
     * Generate invoice for subscription
     *
     * @param OrganizationSubscription $subscription
     * @return Invoice
     */
    public function generateSubscriptionInvoice(OrganizationSubscription $subscription): Invoice
    {
        DB::beginTransaction();
        
        try {
            $plan = $subscription->plan;
            $amount = $subscription->billing_cycle === 'annual' ? 
                $plan->price_annual : 
                $plan->price_monthly;

            $invoice = Invoice::create([
                'organization_id' => $subscription->organization_id,
                'subscription_plan_id' => $plan->id,
                'invoice_number' => Invoice::generateInvoiceNumber(),
                'type' => 'subscription',
                'subtotal' => $amount,
                'tax' => 0, // Add tax calculation if needed
                'discount' => 0,
                'total' => $amount,
                'status' => 'pending',
                'issue_date' => now(),
                'due_date' => now()->addDays(7),
                'line_items' => [
                    [
                        'description' => "{$plan->name} Plan - " . ucfirst($subscription->billing_cycle),
                        'quantity' => 1,
                        'unit_price' => $amount,
                        'amount' => $amount,
                    ],
                ],
            ]);

            Log::info('Invoice generated', [
                'invoice_id' => $invoice->id,
                'organization_id' => $subscription->organization_id,
                'amount' => $amount,
            ]);

            DB::commit();
            return $invoice;

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed to generate invoice', [
                'subscription_id' => $subscription->id,
                'error' => $e->getMessage(),
            ]);
            throw $e;
        }
    }

    /**
     * Generate invoices for all active subscriptions
     *
     * @return int Number of invoices generated
     */
    public function generateMonthlyInvoices(): int
    {
        $subscriptions = OrganizationSubscription::where('status', 'active')
            ->where('next_billing_date', '<=', now())
            ->get();

        $count = 0;

        foreach ($subscriptions as $subscription) {
            try {
                // Check if invoice already exists for this period
                $existingInvoice = Invoice::where('organization_id', $subscription->organization_id)
                    ->where('type', 'subscription')
                    ->where('issue_date', '>=', $subscription->current_period_start)
                    ->where('status', 'pending')
                    ->exists();

                if (!$existingInvoice) {
                    $this->generateSubscriptionInvoice($subscription);
                    $count++;
                }
            } catch (Exception $e) {
                Log::error('Failed to generate monthly invoice', [
                    'subscription_id' => $subscription->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        return $count;
    }

    /**
     * Process payment for invoice
     *
     * @param Invoice $invoice
     * @param Payment $payment
     * @return bool
     */
    public function processPayment(Invoice $invoice, Payment $payment): bool
    {
        if (!$payment->isCompleted()) {
            return false;
        }

        DB::beginTransaction();
        
        try {
            // Mark invoice as paid
            $invoice->markAsPaid();

            // If this is a subscription invoice, update subscription
            if ($invoice->type === 'subscription') {
                $subscription = $invoice->organization->subscription;
                
                if ($subscription) {
                    // If on trial, convert to active
                    if ($subscription->status === 'trial') {
                        app(SubscriptionService::class)->convertTrialToPaid($subscription);
                    } else {
                        // Renew subscription
                        app(SubscriptionService::class)->renewSubscription($subscription);
                    }

                    // Complete onboarding if payment_pending
                    $organization = $invoice->organization;
                    if ($organization->onboarding_step === 'payment_pending') {
                        $organization->update([
                            'onboarding_step' => 'completed',
                            'onboarding_completed_at' => now(),
                        ]);

                        // Mark first login as completed for org admin
                        $orgAdmin = $organization->users()->where('is_system_admin', false)->first();
                        if ($orgAdmin) {
                            $orgAdmin->update(['first_login_completed' => true]);
                        }

                        Log::info('Onboarding completed after payment', [
                            'organization_id' => $organization->id,
                        ]);

                        // Log activity
                        activity()
                            ->performedOn($organization)
                            ->causedBy($orgAdmin)
                            ->withProperties([
                                'plan' => $subscription->plan->name,
                                'amount' => $payment->amount,
                                'payment_method' => $payment->payment_method,
                            ])
                            ->log('Onboarding completed after successful payment');
                    }
                }
            }

            Log::info('Payment processed', [
                'invoice_id' => $invoice->id,
                'payment_id' => $payment->id,
            ]);

            DB::commit();
            return true;

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Failed to process payment', [
                'invoice_id' => $invoice->id,
                'payment_id' => $payment->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Handle failed payment
     *
     * @param Invoice $invoice
     * @param int $attemptNumber
     * @return void
     */
    public function handleFailedPayment(Invoice $invoice, int $attemptNumber = 1): void
    {
        $invoice->update(['status' => 'failed']);

        $maxAttempts = 3;
        $gracePeriodDays = config('subscription.grace_period_days', 7);

        if ($attemptNumber >= $maxAttempts) {
            // Suspend subscription after max attempts
            $subscription = $invoice->organization->subscription;
            
            if ($subscription && $subscription->status === 'active') {
                app(SubscriptionService::class)->suspendSubscription(
                    $subscription,
                    "Payment failed after {$maxAttempts} attempts"
                );
            }

            Log::warning('Subscription suspended due to failed payments', [
                'organization_id' => $invoice->organization_id,
                'invoice_id' => $invoice->id,
            ]);
        } else {
            // Schedule retry
            Log::info('Payment failed, will retry', [
                'invoice_id' => $invoice->id,
                'attempt' => $attemptNumber,
            ]);
        }
    }

    /**
     * Send invoice email
     *
     * @param Invoice $invoice
     * @return bool
     */
    public function sendInvoiceEmail(Invoice $invoice): bool
    {
        try {
            $organization = $invoice->organization;
            $admins = $organization->admins;

            foreach ($admins as $admin) {
                // TODO: Create invoice email mailable
                // Mail::to($admin->email)->send(new InvoiceGenerated($invoice));
            }

            Log::info('Invoice email sent', [
                'invoice_id' => $invoice->id,
            ]);

            return true;

        } catch (Exception $e) {
            Log::error('Failed to send invoice email', [
                'invoice_id' => $invoice->id,
                'error' => $e->getMessage(),
            ]);
            return false;
        }
    }

    /**
     * Get billing summary for organization
     *
     * @param Organization $organization
     * @return array
     */
    public function getBillingSummary(Organization $organization): array
    {
        $invoices = Invoice::where('organization_id', $organization->id)
            ->orderBy('created_at', 'desc')
            ->limit(12)
            ->get();

        $totalPaid = Invoice::where('organization_id', $organization->id)
            ->where('status', 'paid')
            ->sum('total');

        $totalPending = Invoice::where('organization_id', $organization->id)
            ->where('status', 'pending')
            ->sum('total');

        $lastPayment = Payment::where('organization_id', $organization->id)
            ->where('status', 'completed')
            ->latest()
            ->first();

        return [
            'total_paid' => $totalPaid,
            'total_pending' => $totalPending,
            'invoice_count' => $invoices->count(),
            'last_payment' => $lastPayment ? [
                'amount' => $lastPayment->amount,
                'date' => $lastPayment->completed_at->format('Y-m-d'),
                'method' => $lastPayment->payment_method,
            ] : null,
            'recent_invoices' => $invoices->map(function ($invoice) {
                return [
                    'id' => $invoice->id,
                    'invoice_number' => $invoice->invoice_number,
                    'amount' => $invoice->total,
                    'status' => $invoice->status,
                    'due_date' => $invoice->due_date->format('Y-m-d'),
                    'paid_at' => $invoice->paid_at?->format('Y-m-d'),
                ];
            }),
        ];
    }
}
