<?php

namespace App\Services;

use App\Models\AuditLog;
use App\Jobs\ProcessAuditLog;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Queue;

class AuditService
{
    /**
     * Log an audit event asynchronously for better performance
     */
    public static function log(
        string $eventType,
        string $description,
        ?Model $auditable = null,
        array $oldValues = [],
        array $newValues = [],
        array $metadata = [],
        string $severity = 'low',
        string $category = 'data_access'
    ): void {
        $auditData = [
            'user_id' => Auth::id(),
            'organization_id' => Auth::user()?->organization_id,
            'event_type' => $eventType,
            'auditable_type' => $auditable ? get_class($auditable) : null,
            'auditable_id' => $auditable?->getKey(),
            'description' => $description,
            'ip_address' => request()?->ip(),
            'user_agent' => request()?->userAgent(),
            'url' => request()?->fullUrl(),
            'method' => request()?->method(),
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'metadata' => array_merge($metadata, [
                'session_id' => session()->getId(),
                'timestamp' => now()->toISOString(),
            ]),
            'severity' => $severity,
            'category' => $category,
        ];

        // Use queue for better performance and scalability
        if (config('audit.use_queue', true)) {
            ProcessAuditLog::dispatch($auditData);
        } else {
            AuditLog::create($auditData);
        }
    }

    /**
     * Log authentication events
     */
    public static function logAuth(string $eventType, string $description, array $metadata = []): void
    {
        self::log(
            eventType: $eventType,
            description: $description,
            metadata: $metadata,
            severity: $eventType === 'login_failed' ? 'medium' : 'low',
            category: 'authentication'
        );
    }

    /**
     * Log model CRUD operations
     */
    public static function logModelEvent(
        string $eventType,
        Model $model,
        array $oldValues = [],
        array $newValues = []
    ): void {
        $modelName = class_basename($model);
        $description = match($eventType) {
            'created' => "Created {$modelName} #{$model->getKey()}",
            'updated' => "Updated {$modelName} #{$model->getKey()}",
            'deleted' => "Deleted {$modelName} #{$model->getKey()}",
            'restored' => "Restored {$modelName} #{$model->getKey()}",
            default => "{$eventType} {$modelName} #{$model->getKey()}"
        };

        $severity = match($eventType) {
            'deleted' => 'high',
            'updated' => 'medium',
            default => 'low'
        };

        self::log(
            eventType: $eventType,
            description: $description,
            auditable: $model,
            oldValues: $oldValues,
            newValues: $newValues,
            severity: $severity,
            category: 'data_modification'
        );
    }

    /**
     * Log security events
     */
    public static function logSecurity(string $eventType, string $description, array $metadata = []): void
    {
        self::log(
            eventType: $eventType,
            description: $description,
            metadata: $metadata,
            severity: 'high',
            category: 'security'
        );
    }

    /**
     * Log system admin events
     */
    public static function logSystemAdmin(string $eventType, string $description, array $metadata = []): void
    {
        self::log(
            eventType: $eventType,
            description: $description,
            metadata: $metadata,
            severity: 'medium',
            category: 'system_admin'
        );
    }

    /**
     * Get audit statistics for dashboard
     */
    public static function getStatistics(?int $organizationId = null, int $days = 30): array
    {
        $baseQuery = AuditLog::query()
            ->where('created_at', '>=', now()->subDays($days));

        if ($organizationId) {
            $baseQuery->where('organization_id', $organizationId);
        }

        // Get total count
        $totalActivities = (clone $baseQuery)->count();

        // Get category statistics
        $byCategory = (clone $baseQuery)
            ->selectRaw('category, COUNT(*) as count')
            ->groupBy('category')
            ->pluck('count', 'category')
            ->toArray();

        // Get severity statistics
        $bySeverity = (clone $baseQuery)
            ->selectRaw('severity, COUNT(*) as count')
            ->groupBy('severity')
            ->pluck('count', 'severity')
            ->toArray();

        // Get recent activities
        $recentActivities = (clone $baseQuery)
            ->with(['user', 'organization'])
            ->latest()
            ->limit(10)
            ->get();

        // Get top users
        $topUsers = (clone $baseQuery)
            ->selectRaw('user_id, COUNT(*) as count')
            ->whereNotNull('user_id')
            ->groupBy('user_id')
            ->orderByDesc('count')
            ->limit(5)
            ->get()
            ->load('user');

        return [
            'total_activities' => $totalActivities,
            'by_category' => $byCategory,
            'by_severity' => $bySeverity,
            'recent_activities' => $recentActivities,
            'top_users' => $topUsers,
        ];
    }
}