<?php

namespace App\Services;

use App\Models\Employee;
use App\Models\Department;
use App\Models\Attendance;
use App\Models\LeaveRequest;
use App\Models\Payroll;
use App\Models\PerformanceReview;
use Carbon\Carbon;

class AnalyticsService
{
    /**
     * Get real-time attendance metrics for today
     */
    public function getTodayAttendanceMetrics(): array
    {
        $today = today();
        $totalActiveEmployees = Employee::where('status', 'active')->count();
        
        $todayStats = Attendance::where('date', $today)
            ->selectRaw('
                COUNT(*) as total_records,
                SUM(CASE WHEN status = "present" THEN 1 ELSE 0 END) as present,
                SUM(CASE WHEN status = "late" THEN 1 ELSE 0 END) as late,
                SUM(CASE WHEN status = "absent" THEN 1 ELSE 0 END) as absent,
                SUM(CASE WHEN status = "on_leave" THEN 1 ELSE 0 END) as on_leave,
                SUM(CASE WHEN status = "half_day" THEN 1 ELSE 0 END) as half_day
            ')
            ->first();

        $attendanceRate = $totalActiveEmployees > 0 
            ? round((($todayStats->present + $todayStats->late) / $totalActiveEmployees) * 100, 1)
            : 0;

        return [
            'total_employees' => $totalActiveEmployees,
            'today_present' => $todayStats->present ?? 0,
            'today_late' => $todayStats->late ?? 0,
            'today_absent' => $todayStats->absent ?? 0,
            'today_on_leave' => $todayStats->on_leave ?? 0,
            'today_half_day' => $todayStats->half_day ?? 0,
            'attendance_rate' => $attendanceRate,
            'not_marked' => $totalActiveEmployees - ($todayStats->total_records ?? 0)
        ];
    }

    /**
     * Get leave analytics and metrics
     */
    public function getLeaveMetrics(): array
    {
        $currentMonth = now();
        
        return [
            'pending_requests' => LeaveRequest::where('status', 'pending')->count(),
            'approved_this_month' => LeaveRequest::where('status', 'approved')
                ->whereMonth('created_at', $currentMonth->month)
                ->whereYear('created_at', $currentMonth->year)
                ->count(),
            'employees_on_leave_today' => LeaveRequest::where('status', 'approved')
                ->where('start_date', '<=', today())
                ->where('end_date', '>=', today())
                ->count(),
            'total_leave_days_this_month' => LeaveRequest::where('status', 'approved')
                ->whereMonth('start_date', $currentMonth->month)
                ->whereYear('start_date', $currentMonth->year)
                ->sum('number_of_days'),
            'leave_utilization_rate' => $this->calculateLeaveUtilizationRate()
        ];
    }

    /**
     * Get payroll analytics and cost metrics
     */
    public function getPayrollMetrics(): array
    {
        $currentMonth = now()->month;
        $currentYear = now()->year;
        
        $currentPayroll = Payroll::where('month', $currentMonth)
            ->where('year', $currentYear)
            ->selectRaw('
                COUNT(*) as total_employees,
                SUM(gross_salary) as total_gross,
                SUM(net_salary) as total_net,
                SUM(total_deductions) as total_deductions,
                SUM(CASE WHEN status = "paid" THEN 1 ELSE 0 END) as paid_count,
                SUM(CASE WHEN status = "processed" THEN 1 ELSE 0 END) as processed_count,
                AVG(net_salary) as avg_net_salary
            ')
            ->first();

        return [
            'current_month_employees' => $currentPayroll->total_employees ?? 0,
            'total_gross_salary' => $currentPayroll->total_gross ?? 0,
            'total_net_salary' => $currentPayroll->total_net ?? 0,
            'total_deductions' => $currentPayroll->total_deductions ?? 0,
            'paid_employees' => $currentPayroll->paid_count ?? 0,
            'processed_employees' => $currentPayroll->processed_count ?? 0,
            'avg_salary' => round($currentPayroll->avg_net_salary ?? 0, 0),
            'payroll_completion_rate' => $currentPayroll->total_employees > 0 
                ? round(($currentPayroll->paid_count / $currentPayroll->total_employees) * 100, 1)
                : 0
        ];
    }

    /**
     * Get performance review metrics
     */
    public function getPerformanceMetrics(): array
    {
        $currentYear = now()->year;
        
        $performanceStats = PerformanceReview::whereYear('review_date', $currentYear)
            ->selectRaw('
                COUNT(*) as total_reviews,
                AVG(overall_rating) as avg_rating,
                SUM(CASE WHEN status = "completed" THEN 1 ELSE 0 END) as completed_reviews,
                SUM(CASE WHEN due_date < NOW() AND status != "completed" THEN 1 ELSE 0 END) as overdue_reviews
            ')
            ->first();

        return [
            'total_reviews_this_year' => $performanceStats->total_reviews ?? 0,
            'average_rating' => round($performanceStats->avg_rating ?? 0, 1),
            'completed_reviews' => $performanceStats->completed_reviews ?? 0,
            'overdue_reviews' => $performanceStats->overdue_reviews ?? 0,
            'completion_rate' => $performanceStats->total_reviews > 0 
                ? round(($performanceStats->completed_reviews / $performanceStats->total_reviews) * 100, 1)
                : 0
        ];
    }

    /**
     * Get department analytics with employee distribution
     */
    public function getDepartmentAnalytics(): array
    {
        return Department::withCount(['employees' => function ($query) {
                $query->where('status', 'active');
            }])
            ->where('active', true)
            ->orderBy('employees_count', 'desc')
            ->take(8)
            ->get()
            ->map(function ($dept) {
                return [
                    'name' => $dept->name,
                    'employee_count' => $dept->employees_count,
                    'color' => $dept->color ?? $this->generateDepartmentColor($dept->id)
                ];
            })
            ->toArray();
    }

    /**
     * Get attendance trends for the last 6 months
     */
    public function getAttendanceTrends(): array
    {
        $trends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = now()->subMonths($i);
            $monthData = Attendance::whereYear('date', $month->year)
                ->whereMonth('date', $month->month)
                ->selectRaw('
                    SUM(CASE WHEN status = "present" THEN 1 ELSE 0 END) as present,
                    SUM(CASE WHEN status = "late" THEN 1 ELSE 0 END) as late,
                    SUM(CASE WHEN status = "absent" THEN 1 ELSE 0 END) as absent,
                    SUM(CASE WHEN status = "on_leave" THEN 1 ELSE 0 END) as on_leave
                ')
                ->first();

            $trends[] = [
                'month' => $month->format('M'),
                'present' => $monthData->present ?? 0,
                'late' => $monthData->late ?? 0,
                'absent' => $monthData->absent ?? 0,
                'on_leave' => $monthData->on_leave ?? 0
            ];
        }
        return $trends;
    }

    /**
     * Get leave trends for the last 6 months
     */
    public function getLeaveTrends(): array
    {
        $trends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = now()->subMonths($i);
            $leaveData = LeaveRequest::whereYear('start_date', $month->year)
                ->whereMonth('start_date', $month->month)
                ->selectRaw('
                    COUNT(*) as total_requests,
                    SUM(CASE WHEN status = "approved" THEN number_of_days ELSE 0 END) as approved_days,
                    SUM(CASE WHEN status = "pending" THEN 1 ELSE 0 END) as pending_requests
                ')
                ->first();

            $trends[] = [
                'month' => $month->format('M'),
                'total_requests' => $leaveData->total_requests ?? 0,
                'approved_days' => $leaveData->approved_days ?? 0,
                'pending_requests' => $leaveData->pending_requests ?? 0
            ];
        }
        return $trends;
    }

    /**
     * Get payroll cost trends for the last 6 months
     */
    public function getPayrollTrends(): array
    {
        $trends = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = now()->subMonths($i);
            $payrollData = Payroll::where('month', $month->month)
                ->where('year', $month->year)
                ->selectRaw('
                    COUNT(*) as employee_count,
                    SUM(gross_salary) as total_gross,
                    SUM(net_salary) as total_net,
                    AVG(net_salary) as avg_net
                ')
                ->first();

            $trends[] = [
                'month' => $month->format('M'),
                'employee_count' => $payrollData->employee_count ?? 0,
                'total_gross' => $payrollData->total_gross ?? 0,
                'total_net' => $payrollData->total_net ?? 0,
                'avg_net' => round($payrollData->avg_net ?? 0, 0)
            ];
        }
        return $trends;
    }

    /**
     * Calculate leave utilization rate
     */
    private function calculateLeaveUtilizationRate(): float
    {
        $currentYear = now()->year;
        $totalEmployees = Employee::where('status', 'active')->count();
        
        if ($totalEmployees === 0) return 0;

        $totalLeaveDaysUsed = LeaveRequest::where('status', 'approved')
            ->whereYear('start_date', $currentYear)
            ->sum('number_of_days');

        // Assuming average 21 leave days per employee per year
        $totalAvailableLeaveDays = $totalEmployees * 21;
        
        return $totalAvailableLeaveDays > 0 
            ? round(($totalLeaveDaysUsed / $totalAvailableLeaveDays) * 100, 1)
            : 0;
    }

    /**
     * Generate a color for department if not set
     */
    private function generateDepartmentColor(int $id): string
    {
        $colors = [
            '#3498db', '#e74c3c', '#2ecc71', '#f39c12', 
            '#9b59b6', '#1abc9c', '#34495e', '#e67e22'
        ];
        
        return $colors[$id % count($colors)];
    }
}