<?php

namespace App\Policies;

use App\Models\User;

class UserPolicy
{
    /**
     * Determine whether the user can view any users.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_users');
    }

    /**
     * Determine whether the user can view the user.
     */
    public function view(User $user, User $model): bool
    {
        // Users can view their own profile
        if ($user->id === $model->id) {
            return true;
        }

        // System admins can view any user
        if ($user->isSystemAdmin()) {
            return $user->can('view_users');
        }

        // Org admins cannot view system admins
        if ($model->isSystemAdmin()) {
            return false;
        }

        // Org admins can only view users from their organization
        if ($user->organization_id !== $model->organization_id) {
            return false;
        }

        return $user->can('view_users');
    }

    /**
     * Determine whether the user can create users.
     */
    public function create(User $user): bool
    {
        return $user->can('create_users');
    }

    /**
     * Determine whether the user can update the user.
     */
    public function update(User $user, User $model): bool
    {
        // Users can update their own profile
        if ($user->id === $model->id) {
            return true;
        }

        // System admins can update any user
        if ($user->isSystemAdmin()) {
            return $user->can('edit_users');
        }

        // Org admins cannot update system admins
        if ($model->isSystemAdmin()) {
            return false;
        }

        // Org admins can only update users from their organization
        if ($user->organization_id !== $model->organization_id) {
            return false;
        }

        return $user->can('edit_users');
    }

    /**
     * Determine whether the user can delete the user.
     */
    public function delete(User $user, User $model): bool
    {
        // Prevent deleting own account
        if ($user->id === $model->id) {
            return false;
        }

        // System admins can delete any user (except themselves)
        if ($user->isSystemAdmin()) {
            return $user->can('delete_users');
        }

        // Org admins cannot delete system admins
        if ($model->isSystemAdmin()) {
            return false;
        }

        // Org admins can only delete users from their organization
        if ($user->organization_id !== $model->organization_id) {
            return false;
        }

        return $user->can('delete_users');
    }

    /**
     * Determine whether the user can restore the user.
     */
    public function restore(User $user, User $model): bool
    {
        return $user->can('delete_users');
    }

    /**
     * Determine whether the user can permanently delete the user.
     */
    public function forceDelete(User $user, User $model): bool
    {
        return $user->can('delete_users');
    }

    /**
     * Determine whether the user can manage roles for users.
     */
    public function manageRoles(User $user): bool
    {
        return $user->can('manage_user_roles');
    }
}
