<?php

namespace App\Policies;

use App\Models\PayrollSettings;
use App\Models\User;

class PayrollSettingsPolicy
{
    /**
     * Determine whether the user can view payroll settings.
     */
    public function view(User $user, PayrollSettings $settings): bool
    {
        // Users can only view settings for their organization
        if ($user->organization_id !== $settings->organization_id) {
            return false;
        }

        return $user->can('view_payroll') || $user->can('edit_settings');
    }

    /**
     * Determine whether the user can update payroll settings.
     */
    public function update(User $user, PayrollSettings $settings): bool
    {
        // Users can only update settings for their organization
        if ($user->organization_id !== $settings->organization_id) {
            return false;
        }

        // Only organization admins or users with edit_settings permission
        return $user->hasRole('Organization Admin') || $user->can('edit_settings');
    }

    /**
     * Determine whether the user can reset payroll settings.
     */
    public function reset(User $user, PayrollSettings $settings): bool
    {
        return $this->update($user, $settings);
    }

    /**
     * Determine whether the user can test calculations.
     */
    public function testCalculations(User $user): bool
    {
        return $user->can('view_payroll') || $user->can('edit_settings');
    }
}
