<?php

namespace App\Policies;

use App\Models\Payroll;
use App\Models\User;

class PayrollPolicy
{
    /**
     * Determine whether the user can view any payroll records.
     */
    public function viewAny(User $user): bool
    {
        return $user->can('view_payroll');
    }

    /**
     * Determine whether the user can view the payroll record.
     */
    public function view(User $user, Payroll $payroll): bool
    {
        // Users can view their own payroll records
        if ($user->employee && $user->employee->id === $payroll->employee_id) {
            return true;
        }

        return $user->can('view_payroll');
    }

    /**
     * Determine whether the user can create payroll records.
     */
    public function create(User $user): bool
    {
        return $user->can('create_payroll');
    }

    /**
     * Determine whether the user can update the payroll record.
     */
    public function update(User $user, Payroll $payroll): bool
    {
        // Prevent editing paid payroll
        if ($payroll->status === 'paid') {
            return false;
        }

        return $user->can('edit_payroll');
    }

    /**
     * Determine whether the user can delete the payroll record.
     */
    public function delete(User $user, Payroll $payroll): bool
    {
        // Prevent deleting paid payroll
        if ($payroll->status === 'paid') {
            return false;
        }

        return $user->can('delete_payroll');
    }

    /**
     * Determine whether the user can restore the payroll record.
     */
    public function restore(User $user, Payroll $payroll): bool
    {
        return $user->can('delete_payroll');
    }

    /**
     * Determine whether the user can permanently delete the payroll record.
     */
    public function forceDelete(User $user, Payroll $payroll): bool
    {
        return $user->can('delete_payroll');
    }

    /**
     * Determine whether the user can mark payroll as paid.
     */
    public function markPaid(User $user, Payroll $payroll): bool
    {
        return $user->can('edit_payroll') && $payroll->status === 'processed';
    }

    /**
     * Determine whether the user can create bulk payroll.
     */
    public function createBulk(User $user): bool
    {
        return $user->can('create_payroll');
    }

    /**
     * Determine whether the user can approve payroll.
     */
    public function approve(User $user, Payroll $payroll): bool
    {
        // Only pending approval status can be approved
        if ($payroll->approval_status !== 'pending') {
            return false;
        }

        // Check if user has manage_payroll permission
        if (!$user->can('manage_payroll')) {
            return false;
        }

        // Check if user can approve current level using PayrollApprovalService
        $approvalService = app(\App\Services\PayrollApprovalService::class);
        return $approvalService->canUserApprove($payroll, $user);
    }

    /**
     * Determine whether the user can reject payroll.
     */
    public function reject(User $user, Payroll $payroll): bool
    {
        return $this->approve($user, $payroll);
    }

    /**
     * Determine whether the user can process payroll.
     */
    public function process(User $user, Payroll $payroll): bool
    {
        // Only approved payroll can be processed
        if ($payroll->approval_status !== 'approved') {
            return false;
        }

        // Only draft status can be processed
        if ($payroll->status !== 'draft') {
            return false;
        }

        return $user->can('manage_payroll');
    }

    /**
     * Determine whether the user can print payroll.
     */
    public function print(User $user, Payroll $payroll): bool
    {
        // Users can print their own payroll
        if ($user->employee && $user->employee->id === $payroll->employee_id) {
            return true;
        }

        return $user->can('view_payroll');
    }
}
