<?php

namespace App\Policies;

use App\Models\User;
use App\Models\LearningPath;

class LearningPathPolicy
{
    public function viewAny(User $user)
    {
        return $user->can('view_learning_paths') || 
               $user->can('manage_training');
    }

    public function view(User $user, LearningPath $learningPath)
    {
        // All authenticated users can view active learning paths
        if ($learningPath->is_active) {
            return true;
        }

        // Only authorized users can view inactive paths
        return $user->can('view_inactive_learning_paths') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function create(User $user)
    {
        return $user->can('create_learning_paths') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function update(User $user, LearningPath $learningPath)
    {
        // Creator can update their own paths
        if ($learningPath->created_by === $user->id) {
            return true;
        }

        return $user->can('update_learning_paths') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function delete(User $user, LearningPath $learningPath)
    {
        return $user->can('delete_learning_paths') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }

    public function enroll(User $user, LearningPath $learningPath)
    {
        // Users can enroll themselves if they have an employee record
        if ($user->employee && $learningPath->is_active) {
            return true;
        }

        // Managers can enroll their team members
        return $user->can('enroll_others_learning_paths') ||
               $user->hasRole(['Manager', 'HR Manager', 'Training Manager']);
    }

    public function manageEnrollments(User $user)
    {
        return $user->can('manage_learning_path_enrollments') ||
               $user->can('manage_training') ||
               $user->hasRole(['HR Manager', 'Training Manager']);
    }
}