<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class TrainingProgram extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'title',
        'description',
        'training_category_id',
        'type',
        'delivery_method',
        'duration_hours',
        'cost_per_participant',
        'max_participants',
        'prerequisites',
        'learning_objectives',
        'content_outline',
        'instructor_name',
        'instructor_email',
        'provider_name',
        'difficulty_level',
        'target_roles',
        'is_mandatory',
        'is_active',
        'created_by',
        'organization_id',
    ];

    protected $casts = [
        'cost_per_participant' => 'decimal:2',
        'prerequisites' => 'array',
        'learning_objectives' => 'array',
        'target_roles' => 'array',
        'is_mandatory' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function category()
    {
        return $this->belongsTo(TrainingCategory::class, 'training_category_id');
    }

    public function sessions()
    {
        return $this->hasMany(TrainingSession::class);
    }

    public function materials()
    {
        return $this->hasMany(TrainingMaterial::class);
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function enrollments()
    {
        return $this->hasManyThrough(TrainingEnrollment::class, TrainingSession::class);
    }

    public function skills()
    {
        return $this->belongsToMany(Skill::class, 'training_program_skills')
            ->withPivot('skill_level_improvement', 'is_primary_skill')
            ->withTimestamps();
    }

    public function recommendations()
    {
        return $this->hasMany(TrainingRecommendation::class);
    }

    public function performanceImpacts()
    {
        return $this->hasMany(TrainingPerformanceImpact::class);
    }

    public function learningPathSteps()
    {
        return $this->hasMany(LearningPathStep::class);
    }

    public function learningPaths()
    {
        return $this->belongsToMany(LearningPath::class, 'learning_path_steps')
            ->withPivot('step_order', 'is_required', 'prerequisites')
            ->orderBy('learning_path_steps.step_order');
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeMandatory($query)
    {
        return $query->where('is_mandatory', true);
    }

    public function getFormattedDurationAttribute()
    {
        $hours = $this->duration_hours;
        if ($hours < 1) {
            return ($hours * 60) . ' minutes';
        }
        return $hours . ' hour' . ($hours > 1 ? 's' : '');
    }

    public function getAveragePerformanceImprovementAttribute()
    {
        return $this->performanceImpacts()
            ->selectRaw('AVG(performance_score_after - performance_score_before) as avg_improvement')
            ->value('avg_improvement') ?? 0;
    }

    public function getPositiveImpactRateAttribute()
    {
        $totalImpacts = $this->performanceImpacts()->count();
        if ($totalImpacts === 0) return 0;

        $positiveImpacts = $this->performanceImpacts()
            ->whereRaw('performance_score_after > performance_score_before')
            ->count();

        return round(($positiveImpacts / $totalImpacts) * 100, 2);
    }
}