<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Auth;
use App\Traits\HasTenantScope;

class TrainingPerformanceImpact extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'training_program_id',
        'employee_id',
        'organization_id',
        'performance_review_id',
        'performance_score_before',
        'performance_score_after',
        'training_completed_date',
        'performance_measured_date',
        'skill_improvements',
        'impact_notes',
        'measured_by',
    ];

    protected $casts = [
        'performance_score_before' => 'decimal:2',
        'performance_score_after' => 'decimal:2',
        'skill_improvements' => 'array',
        'training_completed_date' => 'date',
        'performance_measured_date' => 'date',
    ];

    public function trainingProgram()
    {
        return $this->belongsTo(TrainingProgram::class);
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function performanceReview()
    {
        return $this->belongsTo(PerformanceReview::class);
    }

    public function measuredBy()
    {
        return $this->belongsTo(User::class, 'measured_by');
    }

    public function getPerformanceImprovementAttribute()
    {
        if ($this->performance_score_before && $this->performance_score_after) {
            return $this->performance_score_after - $this->performance_score_before;
        }
        return null;
    }

    public function getPerformanceImprovementPercentageAttribute()
    {
        if ($this->performance_score_before && $this->performance_score_after && $this->performance_score_before > 0) {
            return round((($this->performance_score_after - $this->performance_score_before) / $this->performance_score_before) * 100, 2);
        }
        return null;
    }

    public function scopePositiveImpact($query)
    {
        return $query->whereRaw('performance_score_after > performance_score_before');
    }

    public function scopeByTrainingProgram($query, $programId)
    {
        return $query->where('training_program_id', $programId);
    }

    public function scopeByEmployee($query, $employeeId)
    {
        return $query->where('employee_id', $employeeId);
    }

    /**
     * Scope to filter by organization through employee relationship
     */
    public function scopeForOrganization($query, $organizationId)
    {
        return $query->whereHas('employee', function ($q) use ($organizationId) {
            $q->where('organization_id', $organizationId);
        });
    }

    /**
     * Global scope removed - using HasTenantScope trait instead
     */
    protected static function booted()
    {
        // Trait handles global scope
    }
}