<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class TrainingEnrollment extends Model
{
    use HasTenantScope;
    
    protected $fillable = [
        'training_session_id',
        'employee_id',
        'organization_id',
        'enrollment_type',
        'status',
        'enrolled_at',
        'attended_at',
        'completed_at',
        'completion_score',
        'feedback',
        'rating',
        'notes',
        'enrolled_by',
    ];

    protected $casts = [
        'enrolled_at' => 'datetime',
        'attended_at' => 'datetime',
        'completed_at' => 'datetime',
        'completion_score' => 'decimal:2',
    ];

    public function session()
    {
        return $this->belongsTo(TrainingSession::class, 'training_session_id');
    }

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function enrolledBy()
    {
        return $this->belongsTo(User::class, 'enrolled_by');
    }

    public function program()
    {
        return $this->hasOneThrough(TrainingProgram::class, TrainingSession::class, 'id', 'id', 'training_session_id', 'training_program_id');
    }

    public function performanceImpacts()
    {
        return $this->hasMany(TrainingPerformanceImpact::class, 'employee_id', 'employee_id')
            ->whereColumn('training_program_id', 'training_programs.id')
            ->join('training_sessions', 'training_enrollments.training_session_id', '=', 'training_sessions.id')
            ->join('training_programs', 'training_sessions.training_program_id', '=', 'training_programs.id');
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    public function scopeAttended($query)
    {
        return $query->whereIn('status', ['attended', 'completed']);
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'enrolled' => 'info',
            'attended' => 'warning',
            'completed' => 'success',
            'failed' => 'danger',
            'cancelled' => 'secondary',
            'no_show' => 'dark',
            default => 'primary'
        };
    }
}