<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class PerformanceSelfAssessment extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'performance_review_id',
        'assessment_date',
        'period',
        'achievements',
        'challenges_faced',
        'strengths_identified',
        'areas_for_development',
        'learning_goals',
        'career_aspirations',
        'support_needed',
        'overall_satisfaction',
        'additional_feedback',
        'status',
        'submitted_at',
        'organization_id',
    ];

    protected $casts = [
        'assessment_date' => 'date',
        'achievements' => 'array',
        'challenges_faced' => 'array',
        'learning_goals' => 'array',
        'career_aspirations' => 'array',
        'overall_satisfaction' => 'decimal:2',
        'submitted_at' => 'datetime',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function performanceReview()
    {
        return $this->belongsTo(PerformanceReview::class);
    }

    public function scopeSubmitted($query)
    {
        return $query->where('status', 'submitted');
    }

    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeByPeriod($query, $period)
    {
        return $query->where('period', $period);
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'draft' => 'secondary',
            'submitted' => 'primary',
            'reviewed' => 'success',
            default => 'secondary'
        };
    }

    public function submit()
    {
        $this->update([
            'status' => 'submitted',
            'submitted_at' => now(),
        ]);
    }

    public function markAsReviewed()
    {
        $this->update(['status' => 'reviewed']);
    }
}