<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class PerformanceReview extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'reviewer_id',
        'review_date',
        'review_period',
        'review_type',
        'is_self_assessment',
        'overall_rating',
        'performance_ratings',
        'goal_achievement_score',
        'strengths',
        'areas_of_improvement',
        'goals',
        'comments',
        'manager_comments',
        'hr_comments',
        'employee_comments',
        'development_areas',
        'status',
        'acknowledged_at',
        'employee_acknowledged_at',
        'due_date',
        'organization_id',
    ];

    protected $casts = [
        'review_date' => 'date',
        'due_date' => 'datetime',
        'overall_rating' => 'decimal:2',
        'goal_achievement_score' => 'decimal:2',
        'performance_ratings' => 'array',
        'development_areas' => 'array',
        'is_self_assessment' => 'boolean',
        'acknowledged_at' => 'datetime',
        'employee_acknowledged_at' => 'datetime',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function reviewer()
    {
        return $this->belongsTo(User::class, 'reviewer_id');
    }

    public function goals()
    {
        return $this->hasMany(PerformanceGoal::class);
    }

    public function selfAssessment()
    {
        return $this->hasOne(PerformanceSelfAssessment::class);
    }

    public function competencyRatings()
    {
        return $this->hasMany(PerformanceCompetencyRating::class);
    }

    public function calculateAverageRating()
    {
        if ($this->performance_ratings && is_array($this->performance_ratings)) {
            $ratings = array_filter($this->performance_ratings);
            return count($ratings) > 0 ? round(array_sum($ratings) / count($ratings), 2) : 0;
        }
        return $this->overall_rating ?? 0;
    }

    public function calculateGoalAchievementScore()
    {
        $activeGoals = $this->goals()->where('status', '!=', 'cancelled')->get();
        
        if ($activeGoals->isEmpty()) {
            return 0;
        }

        $totalWeight = $activeGoals->sum('weight_percentage') ?: 100;
        $weightedScore = 0;

        foreach ($activeGoals as $goal) {
            $weight = $goal->weight_percentage ?: (100 / $activeGoals->count());
            $achievement = $goal->calculateAchievementPercentage();
            $weightedScore += ($achievement * $weight / 100);
        }

        return round($weightedScore * $totalWeight / 100, 2);
    }

    public function getReviewTypeColorAttribute()
    {
        return match($this->review_type) {
            'annual' => 'primary',
            'mid_year' => 'info',
            'quarterly' => 'success',
            'probation' => 'warning',
            'project_based' => 'secondary',
            default => 'primary'
        };
    }

    public function scopeSelfAssessments($query)
    {
        return $query->where('is_self_assessment', true);
    }

    public function scopeManagerReviews($query)
    {
        return $query->where('is_self_assessment', false);
    }

    public function scopeByReviewType($query, $type)
    {
        return $query->where('review_type', $type);
    }

    public function isOverdue()
    {
        return $this->due_date && $this->due_date < now() && $this->status !== 'completed';
    }

    public function trainingImpacts()
    {
        return $this->hasMany(TrainingPerformanceImpact::class);
    }

    public function getRecommendedTrainingsAttribute()
    {
        // Get training recommendations based on performance gaps
        $developmentAreas = $this->development_areas ?? [];
        return TrainingProgram::whereJsonContains('learning_objectives', $developmentAreas)
            ->where('is_active', true)
            ->limit(5)
            ->get();
    }

    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}
