<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class PerformanceGoal extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'performance_review_id',
        'created_by',
        'title',
        'description',
        'type',
        'priority',
        'start_date',
        'target_date',
        'completion_date',
        'status',
        'weight_percentage',
        'target_value',
        'achieved_value',
        'measurement_unit',
        'success_criteria',
        'milestones',
        'progress_percentage',
        'notes',
        'organization_id',
    ];

    protected $casts = [
        'start_date' => 'date',
        'target_date' => 'date',
        'completion_date' => 'date',
        'weight_percentage' => 'integer',
        'target_value' => 'decimal:2',
        'achieved_value' => 'decimal:2',
        'progress_percentage' => 'decimal:2',
        'milestones' => 'array',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function performanceReview()
    {
        return $this->belongsTo(PerformanceReview::class);
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function progressUpdates()
    {
        return $this->hasMany(PerformanceGoalProgress::class)->orderBy('update_date', 'desc');
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeOverdue($query)
    {
        return $query->where('status', '!=', 'completed')
                    ->where('target_date', '<', now());
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeByPriority($query, $priority)
    {
        return $query->where('priority', $priority);
    }

    public function isOverdue()
    {
        return $this->status !== 'completed' && $this->target_date < now();
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'draft' => 'secondary',
            'active' => 'primary',
            'completed' => 'success',
            'cancelled' => 'danger',
            'overdue' => 'warning',
            default => 'secondary'
        };
    }

    public function getPriorityColorAttribute()
    {
        return match($this->priority) {
            'low' => 'info',
            'medium' => 'primary',
            'high' => 'warning',
            'critical' => 'danger',
            default => 'primary'
        };
    }

    public function calculateAchievementPercentage()
    {
        if (!$this->target_value || $this->target_value == 0) {
            return $this->progress_percentage;
        }

        return min(100, ($this->achieved_value / $this->target_value) * 100);
    }

    public function updateProgress($percentage, $achievedValue = null, $notes = null, $updatedBy = null)
    {
        $this->update([
            'progress_percentage' => $percentage,
            'achieved_value' => $achievedValue ?? $this->achieved_value,
        ]);

        if ($notes) {
            $this->progressUpdates()->create([
                'updated_by' => $updatedBy ?? auth()->id(),
                'progress_percentage' => $percentage,
                'achieved_value' => $achievedValue,
                'update_notes' => $notes,
                'update_date' => now(),
            ]);
        }

        // Auto-complete if 100% achieved
        if ($percentage >= 100 && $this->status !== 'completed') {
            $this->update([
                'status' => 'completed',
                'completion_date' => now(),
            ]);
        }
    }
}