<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\HasTenantScope;

class PayrollSettings extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'organization_id',
        'paye_bands',
        'personal_relief',
        'insurance_relief_rate',
        'insurance_relief_max',
        'pension_relief_max',
        'pension_relief_rate',
        'shif_rate',
        'shif_min_contribution',
        'shif_tax_deductible',
        'nssf_employee_rate',
        'nssf_employer_rate',
        'nssf_lower_limit',
        'nssf_upper_limit',
        'nssf_max_employee_contribution',
        'nssf_max_employer_contribution',
        'nssf_min_contribution',
        'housing_levy_employee_rate',
        'housing_levy_employer_rate',
        'housing_levy_tax_deductible',
        'housing_levy_no_max_cap',
        'default_house_allowance_rate',
        'default_transport_allowance_rate',
        'max_transport_allowance',
        'default_medical_allowance',
        'default_monthly_working_days',
        'default_weekly_working_days',
        'default_hours_per_day',
        'kra_submission_deadline',
        'shif_submission_deadline',
        'nssf_submission_deadline',
        'housing_levy_submission_deadline',
        'config_version',
        'last_regulatory_update',
        'regulatory_change_log',
    ];

    protected $casts = [
        'paye_bands' => 'array',
        'personal_relief' => 'decimal:2',
        'insurance_relief_rate' => 'decimal:4',
        'insurance_relief_max' => 'decimal:2',
        'pension_relief_max' => 'decimal:2',
        'pension_relief_rate' => 'decimal:4',
        'shif_rate' => 'decimal:4',
        'shif_min_contribution' => 'decimal:2',
        'shif_tax_deductible' => 'boolean',
        'nssf_employee_rate' => 'decimal:4',
        'nssf_employer_rate' => 'decimal:4',
        'nssf_lower_limit' => 'decimal:2',
        'nssf_upper_limit' => 'decimal:2',
        'nssf_max_employee_contribution' => 'decimal:2',
        'nssf_max_employer_contribution' => 'decimal:2',
        'nssf_min_contribution' => 'decimal:2',
        'housing_levy_employee_rate' => 'decimal:4',
        'housing_levy_employer_rate' => 'decimal:4',
        'housing_levy_tax_deductible' => 'boolean',
        'housing_levy_no_max_cap' => 'boolean',
        'default_house_allowance_rate' => 'decimal:4',
        'default_transport_allowance_rate' => 'decimal:4',
        'max_transport_allowance' => 'decimal:2',
        'default_medical_allowance' => 'decimal:2',
        'last_regulatory_update' => 'datetime',
        'regulatory_change_log' => 'array',
    ];

    /**
     * Get the organization that owns the payroll settings.
     */
    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get the default PAYE tax bands for Kenya (2026 rates).
     */
    public static function getDefaultPayeBands(): array
    {
        return [
            ['min' => 0, 'max' => 24000, 'rate' => 0.10],
            ['min' => 24001, 'max' => 32333, 'rate' => 0.25],
            ['min' => 32334, 'max' => 500000, 'rate' => 0.30],
            ['min' => 500001, 'max' => 800000, 'rate' => 0.32],
            ['min' => 800001, 'max' => null, 'rate' => 0.35],
        ];
    }

    /**
     * Get or create payroll settings for an organization.
     */
    public static function getForOrganization(int $organizationId): self
    {
        return static::firstOrCreate(
            ['organization_id' => $organizationId],
            [
                'paye_bands' => static::getDefaultPayeBands(),
                'config_version' => '2026.02',
                'last_regulatory_update' => now(),
            ]
        );
    }

    /**
     * Calculate PAYE tax for a given taxable income.
     */
    public function calculatePayeTax(float $taxableIncome): float
    {
        $tax = 0;
        $bands = $this->paye_bands ?? static::getDefaultPayeBands();

        foreach ($bands as $band) {
            $min = $band['min'];
            $max = $band['max'] ?? PHP_FLOAT_MAX;
            $rate = $band['rate'];

            if ($taxableIncome > $min) {
                $taxableInBand = min($taxableIncome, $max) - $min;
                $tax += $taxableInBand * $rate;
            }

            if ($taxableIncome <= $max) {
                break;
            }
        }

        return round($tax, 2);
    }

    /**
     * Calculate SHIF contribution.
     */
    public function calculateShifContribution(float $grossSalary): float
    {
        $contribution = $grossSalary * $this->shif_rate;
        return max($contribution, $this->shif_min_contribution);
    }

    /**
     * Calculate NSSF contribution.
     */
    public function calculateNssfContribution(float $grossSalary, bool $isEmployee = true): float
    {
        $rate = $isEmployee ? $this->nssf_employee_rate : $this->nssf_employer_rate;
        $maxContribution = $isEmployee ? $this->nssf_max_employee_contribution : $this->nssf_max_employer_contribution;
        
        $pensionablePay = min($grossSalary, $this->nssf_upper_limit);
        $contribution = $pensionablePay * $rate;
        
        return min($contribution, $maxContribution);
    }

    /**
     * Calculate Housing Levy contribution.
     */
    public function calculateHousingLevyContribution(float $grossSalary, bool $isEmployee = true): float
    {
        $rate = $isEmployee ? $this->housing_levy_employee_rate : $this->housing_levy_employer_rate;
        return $grossSalary * $rate;
    }
}