<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Services\KenyaTaxCalculator;
use App\Traits\HasTenantScope;

class Payroll extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'employment_type',
        'contract_id',
        'payroll_number',
        'month',
        'year',
        'basic_salary',
        'hourly_rate',
        'daily_rate',
        'hours_worked',
        'regular_hours',
        'overtime_hours',
        'overtime_rate',
        'days_worked',
        'expected_working_days',
        'hourly_pay',
        'daily_pay',
        'contract_basic_pay',
        'house_allowance',
        'transport_allowance',
        'medical_allowance',
        'other_allowances',
        'allowances',
        'bonuses',
        'performance_bonus',
        'merit_increase',
        'commission',
        'overtime_pay',
        'gross_salary',
        'taxable_income',
        'paye_tax',
        'nhif_deduction',
        'nssf_employee',
        'nssf_employer',
        'housing_levy',
        'withholding_tax',
        'withholding_tax_rate',
        'helb_deduction',
        'personal_relief',
        'insurance_relief',
        'pension_relief',
        'loan_deductions',
        'advance_deductions',
        'disciplinary_deductions',
        'other_deductions',
        'total_deductions',
        'net_salary',
        'working_days',
        'present_days',
        'absent_days',
        'leave_days',
        'calculation_breakdown',
        'rate_details',
        'prorated',
        'proration_reason',
        'approval_status',
        'approved_by',
        'approved_at',
        'approval_notes',
        'notes',
        'status',
        'payment_date',
        'payment_method',
        'kra_pin',
        'nhif_number',
        'nssf_number',
        'organization_id',
    ];

    protected $casts = [
        'month' => 'integer',
        'year' => 'integer',
        'basic_salary' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'daily_rate' => 'decimal:2',
        'hours_worked' => 'decimal:2',
        'regular_hours' => 'decimal:2',
        'overtime_hours' => 'decimal:2',
        'overtime_rate' => 'decimal:2',
        'days_worked' => 'integer',
        'expected_working_days' => 'integer',
        'hourly_pay' => 'decimal:2',
        'daily_pay' => 'decimal:2',
        'contract_basic_pay' => 'decimal:2',
        'house_allowance' => 'decimal:2',
        'transport_allowance' => 'decimal:2',
        'medical_allowance' => 'decimal:2',
        'other_allowances' => 'decimal:2',
        'allowances' => 'decimal:2',
        'bonuses' => 'decimal:2',
        'performance_bonus' => 'decimal:2',
        'merit_increase' => 'decimal:2',
        'commission' => 'decimal:2',
        'overtime_pay' => 'decimal:2',
        'gross_salary' => 'decimal:2',
        'taxable_income' => 'decimal:2',
        'paye_tax' => 'decimal:2',
        'nhif_deduction' => 'decimal:2',
        'nssf_employee' => 'decimal:2',
        'nssf_employer' => 'decimal:2',
        'housing_levy' => 'decimal:2',
        'withholding_tax' => 'decimal:2',
        'withholding_tax_rate' => 'decimal:4',
        'helb_deduction' => 'decimal:2',
        'personal_relief' => 'decimal:2',
        'insurance_relief' => 'decimal:2',
        'pension_relief' => 'decimal:2',
        'loan_deductions' => 'decimal:2',
        'advance_deductions' => 'decimal:2',
        'disciplinary_deductions' => 'decimal:2',
        'other_deductions' => 'decimal:2',
        'total_deductions' => 'decimal:2',
        'net_salary' => 'decimal:2',
        'working_days' => 'integer',
        'present_days' => 'integer',
        'absent_days' => 'integer',
        'leave_days' => 'integer',
        'calculation_breakdown' => 'array',
        'rate_details' => 'array',
        'prorated' => 'boolean',
        'approved_at' => 'datetime',
        'payment_date' => 'date',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function contract()
    {
        return $this->belongsTo(Contract::class);
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    public function approvals()
    {
        return $this->hasMany(PayrollApproval::class)->orderBy('level');
    }

    public function calculateGrossSalary()
    {
        $this->allowances = $this->house_allowance + $this->transport_allowance + 
                           $this->medical_allowance + $this->other_allowances;
        
        // Calculate basic pay based on employment type
        $basicPay = $this->calculateBasicPay();
        
        $this->gross_salary = $basicPay + $this->allowances + $this->bonuses + 
                             $this->performance_bonus + $this->merit_increase + 
                             $this->commission + $this->overtime_pay;
        
        return $this->gross_salary;
    }

    public function calculateBasicPay()
    {
        $contract = $this->contract ?? $this->employee->activeContract;
        
        if (!$contract) {
            // Fallback to employee basic salary
            return $this->basic_salary ?? $this->employee->basic_salary ?? 0;
        }

        switch ($contract->employment_type) {
            case 'permanent':
                return $this->calculatePermanentBasicPay($contract);
                
            case 'contract':
            case 'freelance':
                return $this->calculateContractBasicPay($contract);
                
            case 'casual':
                return $this->calculateCasualBasicPay($contract);
                
            case 'intern':
            case 'part_time':
                return $this->calculateHourlyBasicPay($contract);
                
            default:
                return $this->basic_salary ?? 0;
        }
    }

    protected function calculatePermanentBasicPay(Contract $contract)
    {
        // For permanent employees, use monthly salary
        $monthlySalary = $contract->salary ?? $this->basic_salary ?? 0;
        
        // Check if prorated (e.g., started mid-month)
        if ($this->prorated) {
            $workingDaysInMonth = $this->expected_working_days ?: 22; // Default 22 working days
            $actualWorkingDays = $this->present_days ?: $this->days_worked ?: $workingDaysInMonth;
            $monthlySalary = ($monthlySalary / $workingDaysInMonth) * $actualWorkingDays;
        }
        
        $this->contract_basic_pay = $monthlySalary;
        return $monthlySalary;
    }

    protected function calculateContractBasicPay(Contract $contract)
    {
        $totalPay = 0;
        
        if ($contract->payment_frequency === 'hourly' || $this->hours_worked > 0) {
            // Calculate based on hours worked
            $hourlyRate = $contract->effective_hourly_rate;
            $regularPay = $this->regular_hours * $hourlyRate;
            $overtimePay = $this->calculateOvertimePay($contract);
            
            $this->hourly_rate = $hourlyRate;
            $this->hourly_pay = $regularPay;
            $this->overtime_pay = $overtimePay;
            
            $totalPay = $regularPay + $overtimePay;
            
        } elseif ($contract->payment_frequency === 'daily' || $this->days_worked > 0) {
            // Calculate based on days worked
            $dailyRate = $contract->effective_daily_rate;
            $totalPay = $this->days_worked * $dailyRate;
            
            $this->daily_rate = $dailyRate;
            $this->daily_pay = $totalPay;
            
        } else {
            // Monthly contract rate
            $totalPay = $contract->salary ?? 0;
        }
        
        $this->contract_basic_pay = $totalPay;
        return $totalPay;
    }

    protected function calculateCasualBasicPay(Contract $contract)
    {
        // Casual workers are typically paid daily
        $dailyRate = $contract->effective_daily_rate;
        $totalPay = $this->days_worked * $dailyRate;
        
        $this->daily_rate = $dailyRate;
        $this->daily_pay = $totalPay;
        $this->contract_basic_pay = $totalPay;
        
        return $totalPay;
    }

    protected function calculateHourlyBasicPay(Contract $contract)
    {
        // Interns and part-time workers paid hourly
        $hourlyRate = $contract->effective_hourly_rate;
        $regularPay = $this->regular_hours * $hourlyRate;
        $overtimePay = $contract->overtime_eligible ? $this->calculateOvertimePay($contract) : 0;
        
        $this->hourly_rate = $hourlyRate;
        $this->hourly_pay = $regularPay;
        $this->overtime_pay = $overtimePay;
        $this->contract_basic_pay = $regularPay + $overtimePay;
        
        return $regularPay + $overtimePay;
    }

    protected function calculateOvertimePay(Contract $contract)
    {
        if (!$contract->overtime_eligible || $this->overtime_hours <= 0) {
            return 0;
        }
        
        $baseHourlyRate = $contract->effective_hourly_rate;
        $overtimeRate = $baseHourlyRate * $contract->overtime_rate_multiplier;
        
        $this->overtime_rate = $overtimeRate;
        
        return $this->overtime_hours * $overtimeRate;
    }

    public function calculateTotalDeductions()
    {
        $this->total_deductions = $this->paye_tax + $this->nhif_deduction + $this->nssf_employee + 
                                 $this->housing_levy + $this->withholding_tax + $this->helb_deduction + 
                                 $this->loan_deductions + $this->advance_deductions + 
                                 $this->disciplinary_deductions + $this->other_deductions;
        return $this->total_deductions;
    }

    public function calculateNetSalary()
    {
        $this->calculateGrossSalary();
        $this->calculateTotalDeductions();
        $this->net_salary = $this->gross_salary - $this->total_deductions;
        return $this->net_salary;
    }

    /**
     * Calculate complete payroll using Kenya Tax Calculator with employment type support
     */
    public function calculateKenyaPayroll(array $salaryComponents = [])
    {
        $contract = $this->contract ?? $this->employee->activeContract;
        
        // Set employment type from contract
        if ($contract) {
            $this->employment_type = $contract->employment_type;
            $this->contract_id = $contract->id;
        }
        
        // Calculate basic pay first
        $basicPay = $this->calculateBasicPay();
        
        // Use existing values if components not provided
        $components = array_merge([
            'basic_salary' => $basicPay,
            'house_allowance' => $this->house_allowance,
            'transport_allowance' => $this->transport_allowance,
            'medical_allowance' => $this->medical_allowance,
            'other_allowances' => $this->other_allowances,
            'bonuses' => $this->bonuses,
            'performance_bonus' => $this->performance_bonus,
            'merit_increase' => $this->merit_increase,
            'commission' => $this->commission,
            'overtime_pay' => $this->overtime_pay,
            'helb_deduction' => $this->helb_deduction,
            'loan_deductions' => $this->loan_deductions,
            'advance_deductions' => $this->advance_deductions,
            'disciplinary_deductions' => $this->disciplinary_deductions,
            'other_deductions' => $this->other_deductions,
        ], $salaryComponents);

        // Calculate using Kenya Tax Calculator
        $calculations = KenyaTaxCalculator::calculatePayroll($components);

        // Handle withholding tax for contractors
        if ($contract && $contract->shouldApplyWithholdingTax()) {
            $this->withholding_tax_rate = $contract->withholding_tax_rate;
            $this->withholding_tax = $calculations['gross_salary'] * $this->withholding_tax_rate;
            $calculations['withholding_tax'] = $this->withholding_tax;
            $calculations['total_deductions'] += $this->withholding_tax;
            $calculations['net_salary'] -= $this->withholding_tax;
        }

        // Apply statutory deductions based on contract eligibility
        if ($contract && !$contract->statutory_deductions_applicable) {
            // For contractors/freelancers who are not eligible for statutory benefits
            $calculations['nhif_deduction'] = 0;
            $calculations['nssf_employee'] = 0;
            $calculations['housing_levy'] = 0;
            
            // Recalculate totals
            $calculations['total_deductions'] = $calculations['paye_tax'] + 
                                              $this->withholding_tax + 
                                              $calculations['helb_deduction'] + 
                                              $calculations['loan_deductions'] + 
                                              $calculations['advance_deductions'] + 
                                              $calculations['disciplinary_deductions'] + 
                                              $calculations['other_deductions'];
            
            $calculations['net_salary'] = $calculations['gross_salary'] - $calculations['total_deductions'];
        }

        // Store calculation breakdown for audit trail
        $this->calculation_breakdown = [
            'employment_type' => $this->employment_type,
            'contract_id' => $this->contract_id,
            'calculation_date' => now()->toISOString(),
            'basic_pay_method' => $this->getBasicPayMethod(),
            'rates_used' => $this->getRatesUsed($contract),
            'hours_breakdown' => $this->getHoursBreakdown(),
            'statutory_eligibility' => $contract?->statutory_deductions_applicable ?? true,
            'withholding_tax_applied' => $contract?->shouldApplyWithholdingTax() ?? false,
        ];

        // Update model with calculated values
        foreach ($calculations as $field => $value) {
            if (in_array($field, $this->fillable)) {
                $this->$field = $value;
            }
        }

        return $calculations;
    }

    protected function getBasicPayMethod()
    {
        $contract = $this->contract ?? $this->employee->activeContract;
        
        if (!$contract) return 'employee_basic_salary';
        
        return match($contract->employment_type) {
            'permanent' => 'monthly_salary',
            'contract', 'freelance' => $contract->payment_frequency,
            'casual' => 'daily_rate',
            'intern', 'part_time' => 'hourly_rate',
            default => 'monthly_salary'
        };
    }

    protected function getRatesUsed($contract)
    {
        if (!$contract) return [];
        
        return [
            'hourly_rate' => $contract->effective_hourly_rate,
            'daily_rate' => $contract->effective_daily_rate,
            'overtime_rate' => $contract->overtime_rate,
            'overtime_multiplier' => $contract->overtime_rate_multiplier,
            'withholding_tax_rate' => $contract->withholding_tax_rate,
        ];
    }

    protected function getHoursBreakdown()
    {
        return [
            'total_hours_worked' => $this->hours_worked,
            'regular_hours' => $this->regular_hours,
            'overtime_hours' => $this->overtime_hours,
            'days_worked' => $this->days_worked,
            'expected_working_days' => $this->expected_working_days,
        ];
    }

    /**
     * Load attendance data for payroll calculation
     */
    public function loadAttendanceData($year, $month)
    {
        $attendances = $this->employee->attendances()
            ->byMonth($year, $month)
            ->approved()
            ->notInPayroll()
            ->get();

        $totalHours = 0;
        $totalRegularHours = 0;
        $totalOvertimeHours = 0;
        $totalDaysWorked = 0;

        foreach ($attendances as $attendance) {
            // Ensure attendance hours are calculated
            if (!$attendance->regular_hours) {
                $attendance->calculateDetailedHours();
            }

            $totalHours += $attendance->worked_hours;
            $totalRegularHours += $attendance->regular_hours;
            $totalOvertimeHours += $attendance->overtime_hours;
            $totalDaysWorked++;

            // Mark attendance as included in this payroll
            $attendance->markAsIncludedInPayroll($this);
        }

        $this->hours_worked = $totalHours;
        $this->regular_hours = $totalRegularHours;
        $this->overtime_hours = $totalOvertimeHours;
        $this->days_worked = $totalDaysWorked;
        $this->present_days = $totalDaysWorked;

        // Calculate expected working days for the month
        $this->expected_working_days = $this->calculateExpectedWorkingDays($year, $month);
        $this->absent_days = max(0, $this->expected_working_days - $this->present_days);

        return $attendances;
    }

    protected function calculateExpectedWorkingDays($year, $month)
    {
        $startDate = Carbon::create($year, $month, 1);
        $endDate = $startDate->copy()->endOfMonth();
        
        $workingDays = 0;
        $current = $startDate->copy();
        
        while ($current <= $endDate) {
            // Skip weekends (Saturday = 6, Sunday = 0)
            if (!in_array($current->dayOfWeek, [0, 6])) {
                $workingDays++;
            }
            $current->addDay();
        }
        
        // TODO: Subtract public holidays
        return $workingDays;
    }

    /**
     * Get statutory deductions summary with employment type consideration
     */
    public function getStatutoryDeductionsAttribute()
    {
        $contract = $this->contract ?? $this->employee->activeContract;
        
        $deductions = [
            'paye_tax' => $this->paye_tax,
            'withholding_tax' => $this->withholding_tax,
        ];
        
        // Only include statutory benefits if eligible
        if (!$contract || $contract->statutory_deductions_applicable) {
            $deductions['nhif_deduction'] = $this->nhif_deduction;
            $deductions['nssf_employee'] = $this->nssf_employee;
            $deductions['housing_levy'] = $this->housing_levy;
        }
        
        $deductions['total'] = array_sum($deductions);
        
        return $deductions;
    }

    /**
     * Get employment type specific pay breakdown
     */
    public function getPayBreakdownAttribute()
    {
        $breakdown = [
            'employment_type' => $this->employment_type,
            'basic_pay_components' => [],
            'allowances' => $this->allowances_breakdown,
            'bonuses' => [
                'regular_bonus' => $this->bonuses,
                'performance_bonus' => $this->performance_bonus,
                'merit_increase' => $this->merit_increase,
                'commission' => $this->commission,
            ],
            'deductions' => $this->statutory_deductions,
        ];

        // Add employment type specific components
        switch ($this->employment_type) {
            case 'contract':
            case 'freelance':
                $breakdown['basic_pay_components'] = [
                    'hourly_rate' => $this->hourly_rate,
                    'hours_worked' => $this->hours_worked,
                    'hourly_pay' => $this->hourly_pay,
                    'overtime_rate' => $this->overtime_rate,
                    'overtime_hours' => $this->overtime_hours,
                    'overtime_pay' => $this->overtime_pay,
                ];
                break;
                
            case 'casual':
                $breakdown['basic_pay_components'] = [
                    'daily_rate' => $this->daily_rate,
                    'days_worked' => $this->days_worked,
                    'daily_pay' => $this->daily_pay,
                ];
                break;
                
            case 'permanent':
            default:
                $breakdown['basic_pay_components'] = [
                    'monthly_salary' => $this->basic_salary,
                    'prorated' => $this->prorated,
                    'working_days' => $this->working_days,
                    'present_days' => $this->present_days,
                ];
                break;
        }

        return $breakdown;
    }

    /**
     * Get allowances breakdown
     */
    public function getAllowancesBreakdownAttribute()
    {
        return [
            'house_allowance' => $this->house_allowance,
            'transport_allowance' => $this->transport_allowance,
            'medical_allowance' => $this->medical_allowance,
            'other_allowances' => $this->other_allowances,
            'total' => $this->allowances,
        ];
    }

    public function scopeByMonth($query, $year, $month)
    {
        return $query->where('year', $year)->where('month', $month);
    }

    public function scopeProcessed($query)
    {
        return $query->where('status', 'processed');
    }

    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    public function scopeByEmploymentType($query, $type)
    {
        return $query->where('employment_type', $type);
    }

    public function scopePendingApproval($query)
    {
        return $query->where('approval_status', 'pending');
    }

    public function scopeApproved($query)
    {
        return $query->where('approval_status', 'approved');
    }

    public function approve($approverId, $notes = null)
    {
        $this->update([
            'approval_status' => 'approved',
            'approved_by' => $approverId,
            'approved_at' => now(),
            'approval_notes' => $notes,
        ]);
    }

    public function reject($approverId, $notes = null)
    {
        $this->update([
            'approval_status' => 'rejected',
            'approved_by' => $approverId,
            'approved_at' => now(),
            'approval_notes' => $notes,
        ]);
    }

    public function getEmploymentTypeDisplayAttribute()
    {
        return match($this->employment_type) {
            'permanent' => 'Permanent Employee',
            'contract' => 'Contract Employee',
            'casual' => 'Casual Worker',
            'intern' => 'Intern',
            'freelance' => 'Freelancer',
            'part_time' => 'Part-time Employee',
            default => ucfirst($this->employment_type)
        };
    }

    public function getApprovalStatusColorAttribute()
    {
        return match($this->approval_status) {
            'approved' => 'success',
            'rejected' => 'danger',
            'pending' => 'warning',
            default => 'secondary'
        };
    }

    public function getTotalEarningsAttribute()
    {
        return $this->contract_basic_pay + $this->allowances + $this->bonuses + 
               $this->performance_bonus + $this->merit_increase + $this->commission + $this->overtime_pay;
    }

    public function getNetPayAttribute()
    {
        return $this->total_earnings - $this->total_deductions;
    }

    /**
     * Create payroll from attendance data
     */
    public static function createFromAttendance(Employee $employee, $year, $month)
    {
        $contract = $employee->activeContract;
        
        if (!$contract) {
            throw new \Exception("Employee {$employee->full_name} does not have an active contract");
        }

        $payroll = new static([
            'employee_id' => $employee->id,
            'employment_type' => $contract->employment_type,
            'contract_id' => $contract->id,
            'month' => $month,
            'year' => $year,
            'payroll_number' => static::generatePayrollNumber($employee, $year, $month),
            'organization_id' => $employee->organization_id,
        ]);

        // Load attendance data
        $payroll->loadAttendanceData($year, $month);
        
        // Calculate payroll
        $payroll->calculateKenyaPayroll();
        
        $payroll->save();
        
        return $payroll;
    }

    public static function generatePayrollNumber(Employee $employee, $year, $month)
    {
        return sprintf('PAY-%s-%04d%02d-%04d', 
            $employee->employee_code, 
            $year, 
            $month, 
            static::where('year', $year)->where('month', $month)->count() + 1
        );
    }
}
