<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Payment extends Model
{
    use HasFactory;

    protected $fillable = [
        'invoice_id',
        'organization_id',
        'amount',
        'currency',
        'payment_method',
        'status',
        'mpesa_checkout_request_id',
        'mpesa_merchant_request_id',
        'mpesa_receipt_number',
        'mpesa_phone_number',
        'mpesa_amount',
        'mpesa_result_code',
        'mpesa_result_desc',
        'transaction_id',
        'reference_number',
        'gateway_response',
        'initiated_at',
        'completed_at',
        'failed_at',
        'notes',
        'ip_address',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'mpesa_amount' => 'decimal:2',
        'gateway_response' => 'array',
        'initiated_at' => 'datetime',
        'completed_at' => 'datetime',
        'failed_at' => 'datetime',
    ];

    /**
     * Get the invoice that owns the payment
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * Get the organization that owns the payment
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Check if payment is completed
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if payment is pending
     */
    public function isPending(): bool
    {
        return in_array($this->status, ['pending', 'processing']);
    }

    /**
     * Check if payment failed
     */
    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Mark payment as completed
     */
    public function markAsCompleted(array $data = [])
    {
        $this->update(array_merge([
            'status' => 'completed',
            'completed_at' => now(),
        ], $data));

        // Mark invoice as paid if exists
        if ($this->invoice) {
            $this->invoice->markAsPaid();
        }
    }

    /**
     * Mark payment as failed
     */
    public function markAsFailed(string $reason = null)
    {
        $this->update([
            'status' => 'failed',
            'failed_at' => now(),
            'notes' => $reason,
        ]);

        // Mark invoice as failed if exists
        if ($this->invoice) {
            $this->invoice->markAsFailed();
        }
    }

    /**
     * Scope for completed payments
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Scope for pending payments
     */
    public function scopePending($query)
    {
        return $query->whereIn('status', ['pending', 'processing']);
    }

    /**
     * Scope for M-Pesa payments
     */
    public function scopeMpesa($query)
    {
        return $query->where('payment_method', 'mpesa');
    }
}
