<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class OrganizationSubscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'organization_id',
        'subscription_plan_id',
        'status',
        'billing_cycle',
        'trial_starts_at',
        'trial_ends_at',
        'trial_extended',
        'current_period_start',
        'current_period_end',
        'next_billing_date',
        'cancel_at_period_end',
        'cancelled_at',
        'cancellation_reason',
        'activated_at',
        'suspended_at',
        'suspension_reason',
    ];

    protected $casts = [
        'trial_starts_at' => 'datetime',
        'trial_ends_at' => 'datetime',
        'trial_extended' => 'boolean',
        'current_period_start' => 'datetime',
        'current_period_end' => 'datetime',
        'next_billing_date' => 'datetime',
        'cancel_at_period_end' => 'boolean',
        'cancelled_at' => 'datetime',
        'activated_at' => 'datetime',
        'suspended_at' => 'datetime',
    ];

    /**
     * Get the organization that owns the subscription
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get the subscription plan
     */
    public function plan()
    {
        return $this->belongsTo(SubscriptionPlan::class, 'subscription_plan_id');
    }

    /**
     * Check if subscription is on trial
     */
    public function onTrial(): bool
    {
        return $this->status === 'trial' && 
               $this->trial_ends_at && 
               $this->trial_ends_at->isFuture();
    }

    /**
     * Check if trial has expired
     */
    public function trialExpired(): bool
    {
        return $this->status === 'trial' && 
               $this->trial_ends_at && 
               $this->trial_ends_at->isPast();
    }

    /**
     * Get days remaining in trial
     */
    public function trialDaysRemaining(): int
    {
        if (!$this->onTrial()) {
            return 0;
        }
        
        return max(0, (int) now()->diffInDays($this->trial_ends_at, false));
    }

    /**
     * Check if subscription is active
     */
    public function isActive(): bool
    {
        return in_array($this->status, ['trial', 'active']) && 
               (!$this->onTrial() || !$this->trialExpired());
    }

    /**
     * Check if subscription is suspended
     */
    public function isSuspended(): bool
    {
        return $this->status === 'suspended';
    }

    /**
     * Check if subscription is cancelled
     */
    public function isCancelled(): bool
    {
        return $this->status === 'cancelled';
    }

    /**
     * Activate subscription
     */
    public function activate()
    {
        $this->update([
            'status' => 'active',
            'activated_at' => now(),
            'suspended_at' => null,
            'suspension_reason' => null,
        ]);
    }

    /**
     * Suspend subscription
     */
    public function suspend(string $reason = null)
    {
        $this->update([
            'status' => 'suspended',
            'suspended_at' => now(),
            'suspension_reason' => $reason,
        ]);
    }

    /**
     * Cancel subscription
     */
    public function cancel(string $reason = null, bool $immediately = false)
    {
        if ($immediately) {
            $this->update([
                'status' => 'cancelled',
                'cancelled_at' => now(),
                'cancellation_reason' => $reason,
            ]);
        } else {
            $this->update([
                'cancel_at_period_end' => true,
                'cancellation_reason' => $reason,
            ]);
        }
    }

    /**
     * Scope for active subscriptions
     */
    public function scopeActive($query)
    {
        return $query->whereIn('status', ['trial', 'active']);
    }

    /**
     * Scope for trial subscriptions
     */
    public function scopeTrial($query)
    {
        return $query->where('status', 'trial');
    }

    /**
     * Scope for expired trials
     */
    public function scopeExpiredTrials($query)
    {
        return $query->where('status', 'trial')
                    ->where('trial_ends_at', '<', now());
    }
}
