<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Organization extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'logo',
        'settings',
        'status',
        'subscription_status',
        'trial_ends_at',
        'subscription_ends_at',
        'mail_driver',
        'mail_host',
        'mail_port',
        'mail_username',
        'mail_password',
        'mail_encryption',
        'mail_from_address',
        'mail_from_name',
        'use_custom_mail',
        'onboarding_step',
        'onboarding_completed_at',
        'onboarding_notes',
    ];

    protected $casts = [
        'settings' => 'array',
        'use_custom_mail' => 'boolean',
        'mail_port' => 'integer',
        'trial_ends_at' => 'datetime',
        'subscription_ends_at' => 'datetime',
        'onboarding_completed_at' => 'datetime',
    ];

    protected $hidden = [
        'mail_password',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($organization) {
            if (empty($organization->slug)) {
                $organization->slug = Str::slug($organization->name);
            }
        });
    }

    /**
     * Get all users belonging to this organization
     */
    public function users()
    {
        return $this->hasMany(User::class);
    }

    /**
     * Get all employees belonging to this organization
     */
    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    /**
     * Get all departments belonging to this organization
     */
    public function departments()
    {
        return $this->hasMany(Department::class);
    }

    /**
     * Get the theme settings for this organization
     */
    public function themeSetting()
    {
        return $this->hasOne(ThemeSetting::class);
    }

    /**
     * Get the current subscription
     */
    public function subscription()
    {
        return $this->hasOne(OrganizationSubscription::class)->latest();
    }

    /**
     * Get all subscriptions (history)
     */
    public function subscriptions()
    {
        return $this->hasMany(OrganizationSubscription::class);
    }

    /**
     * Get all invoices
     */
    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    /**
     * Get all payments
     */
    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get usage records
     */
    public function usageRecords()
    {
        return $this->hasMany(OrganizationUsage::class);
    }

    /**
     * Get organization admin users
     */
    public function admins()
    {
        return $this->users()->role('org_admin');
    }

    /**
     * Check if organization is active
     */
    public function isActive()
    {
        return $this->status === 'active';
    }

    /**
     * Get organization setting
     */
    public function getSetting($key, $default = null)
    {
        return data_get($this->settings, $key, $default);
    }

    /**
     * Set organization setting
     */
    public function setSetting($key, $value)
    {
        $settings = $this->settings ?? [];
        data_set($settings, $key, $value);
        $this->update(['settings' => $settings]);
    }

    /**
     * Check if organization is on trial
     */
    public function onTrial(): bool
    {
        return $this->subscription_status === 'trial' && 
               $this->trial_ends_at && 
               $this->trial_ends_at->isFuture();
    }

    /**
     * Check if subscription is active (including trial)
     */
    public function hasActiveSubscription(): bool
    {
        return in_array($this->subscription_status, ['trial', 'active']);
    }

    /**
     * Get current subscription plan
     */
    public function currentPlan()
    {
        return $this->subscription?->plan;
    }

    /**
     * Check if organization can access feature
     */
    public function canAccessFeature(string $feature): bool
    {
        $plan = $this->currentPlan();
        return $plan && $plan->hasFeature($feature);
    }

    /**
     * Get quota limit for metric
     */
    public function getQuotaLimit(string $metric)
    {
        $plan = $this->currentPlan();
        if (!$plan) {
            return 0;
        }

        return match($metric) {
            'employees' => $plan->max_employees,
            'users' => $plan->max_users,
            'departments' => $plan->max_departments,
            'storage' => $plan->storage_limit_gb,
            'api_calls' => $plan->api_call_limit,
            default => null,
        };
    }

    /**
     * Check if quota is exceeded
     */
    public function isQuotaExceeded(string $metric, int $currentValue): bool
    {
        $limit = $this->getQuotaLimit($metric);
        
        // null means unlimited
        if (is_null($limit)) {
            return false;
        }

        return $currentValue >= $limit;
    }
}