<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;
use App\Traits\HasTenantScope;

class JobOffer extends Model
{
    use HasTenantScope;

    protected $fillable = [
        'application_id',
        'offer_token',
        'position_title',
        'salary',
        'salary_period',
        'start_date',
        'benefits',
        'terms_and_conditions',
        'offer_letter_path',
        'status',
        'sent_at',
        'expires_at',
        'responded_at',
        'candidate_notes',
        'created_by',
        'organization_id',
    ];

    protected $casts = [
        'salary' => 'decimal:2',
        'start_date' => 'date',
        'sent_at' => 'datetime',
        'expires_at' => 'datetime',
        'responded_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($offer) {
            if (empty($offer->offer_token)) {
                $offer->offer_token = Str::random(32);
            }
        });
    }

    /**
     * Get the application that owns the offer
     */
    public function application()
    {
        return $this->belongsTo(Application::class);
    }

    /**
     * Get the organization that owns the offer
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get the user who created the offer
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the offer URL
     */
    public function getOfferUrlAttribute()
    {
        return route('offer.view', $this->offer_token);
    }

    /**
     * Check if offer is pending
     */
    public function isPending()
    {
        return $this->status === 'sent' && 
               ($this->expires_at === null || $this->expires_at->isFuture());
    }

    /**
     * Check if offer is expired
     */
    public function isExpired()
    {
        return $this->status === 'sent' && 
               $this->expires_at !== null && 
               $this->expires_at->isPast();
    }

    /**
     * Check if offer is accepted
     */
    public function isAccepted()
    {
        return $this->status === 'accepted';
    }

    /**
     * Check if offer is declined
     */
    public function isDeclined()
    {
        return $this->status === 'declined';
    }

    /**
     * Get formatted salary
     */
    public function getFormattedSalaryAttribute()
    {
        $amount = number_format($this->salary, 2);
        return match($this->salary_period) {
            'hourly' => '$' . $amount . '/hour',
            'monthly' => '$' . $amount . '/month',
            'annually' => '$' . $amount . '/year',
            default => '$' . $amount,
        };
    }
}
