<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class Interview extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'application_id',
        'scheduled_at',
        'location',
        'meeting_link',
        'interviewer_id',
        'status',
        'confirmation_token',
        'candidate_response',
        'candidate_responded_at',
        'candidate_notes',
        'total_score',
        'notes',
        'organization_id',
    ];

    protected $casts = [
        'scheduled_at' => 'datetime',
        'candidate_responded_at' => 'datetime',
        'total_score' => 'decimal:2',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($interview) {
            if (empty($interview->confirmation_token)) {
                $interview->confirmation_token = \Illuminate\Support\Str::random(32);
            }
        });
    }

    /**
     * Get the application that owns the interview
     */
    public function application()
    {
        return $this->belongsTo(Application::class);
    }

    /**
     * Get the interviewer
     */
    public function interviewer()
    {
        return $this->belongsTo(User::class, 'interviewer_id');
    }

    /**
     * Get the organization that owns the interview
     */
    public function organization()
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get all questions for this interview
     */
    public function questions()
    {
        return $this->hasMany(InterviewQuestion::class);
    }

    /**
     * Calculate total score from questions
     */
    public function calculateTotalScore()
    {
        $questions = $this->questions()->whereNotNull('score')->get();
        
        if ($questions->isEmpty()) {
            return null;
        }

        $totalScore = $questions->sum('score');
        $maxScore = $questions->sum('max_score');

        return $maxScore > 0 ? ($totalScore / $maxScore) * 100 : 0;
    }

    /**
     * Check if interview is upcoming
     */
    public function isUpcoming()
    {
        return $this->status === 'scheduled' && $this->scheduled_at->isFuture();
    }

    /**
     * Check if interview is completed
     */
    public function isCompleted()
    {
        return $this->status === 'completed';
    }

    /**
     * Scope to upcoming interviews
     */
    public function scopeUpcoming($query)
    {
        return $query->where('status', 'scheduled')
                    ->where('scheduled_at', '>', now());
    }

    /**
     * Scope to completed interviews
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }
}