<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Traits\HasTenantScope;

class Employee extends Model
{
    use HasFactory, SoftDeletes, HasTenantScope;

    protected $fillable = [
        'user_id',
        'employee_code',
        'first_name',
        'last_name',
        'email',
        'phone',
        'mobile',
        'date_of_birth',
        'gender',
        'marital_status',
        'nationality',
        'identification_id',
        'kra_pin',
        'nhif_number',
        'nssf_number',
        'basic_salary',
        'has_helb_loan',
        'helb_monthly_deduction',
        'passport_id',
        'work_permit',
        'visa_no',
        'visa_expire',
        'department_id',
        'position_id',
        'manager_id',
        'coach_id',
        'work_location',
        'work_email',
        'work_phone',
        'joining_date',
        'address',
        'city',
        'state',
        'zip_code',
        'country',
        'emergency_contact_name',
        'emergency_contact_phone',
        'emergency_contact_relation',
        'bank_name',
        'bank_account_number',
        'bank_account_holder',
        'bank_ifsc',
        'photo',
        'notes',
        'status',
        'is_company_admin',
        'organization_id',
    ];

    protected $casts = [
        'date_of_birth' => 'date',
        'visa_expire' => 'date',
        'joining_date' => 'date',
        'basic_salary' => 'decimal:2',
        'helb_monthly_deduction' => 'decimal:2',
        'has_helb_loan' => 'boolean',
        'is_company_admin' => 'boolean',
    ];

    protected $appends = ['full_name'];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    public function position()
    {
        return $this->belongsTo(Position::class);
    }

    public function manager()
    {
        return $this->belongsTo(Employee::class, 'manager_id');
    }

    public function subordinates()
    {
        return $this->hasMany(Employee::class, 'manager_id');
    }

    public function coach()
    {
        return $this->belongsTo(Employee::class, 'coach_id');
    }

    public function contracts()
    {
        return $this->hasMany(Contract::class);
    }

    public function activeContract()
    {
        return $this->hasOne(Contract::class)->where('status', 'active')->latest();
    }

    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    public function leaveRequests()
    {
        return $this->hasMany(LeaveRequest::class);
    }

    public function leaveBalances()
    {
        return $this->hasMany(LeaveBalance::class);
    }

    public function payrolls()
    {
        return $this->hasMany(Payroll::class);
    }

    public function p9Forms()
    {
        return $this->hasMany(P9Form::class);
    }

    public function skills()
    {
        return $this->belongsToMany(Skill::class, 'employee_skills')
            ->withPivot('proficiency_level', 'acquired_date', 'notes')
            ->withTimestamps();
    }

    public function performanceReviews()
    {
        return $this->hasMany(PerformanceReview::class);
    }

    public function trainingEnrollments()
    {
        return $this->hasMany(TrainingEnrollment::class);
    }

    public function completedTrainings()
    {
        return $this->trainingEnrollments()->where('status', 'completed');
    }

    public function learningPathEnrollments()
    {
        return $this->hasMany(LearningPathEnrollment::class);
    }

    public function trainingRecommendations()
    {
        return $this->hasMany(TrainingRecommendation::class);
    }

    public function performanceImpacts()
    {
        return $this->hasMany(TrainingPerformanceImpact::class);
    }

    public function getTrainingCompletionRateAttribute()
    {
        $total = $this->trainingEnrollments()->count();
        if ($total === 0) return 0;
        
        $completed = $this->completedTrainings()->count();
        return round(($completed / $total) * 100, 2);
    }

    public function getPendingRecommendationsAttribute()
    {
        return $this->trainingRecommendations()->where('status', 'pending')->count();
    }

    public function getAveragePerformanceImprovementAttribute()
    {
        return $this->performanceImpacts()
            ->selectRaw('AVG(performance_score_after - performance_score_before) as avg_improvement')
            ->value('avg_improvement') ?? 0;
    }

    public function documents()
    {
        return $this->hasMany(EmployeeDocument::class);
    }

    public function loans()
    {
        return $this->hasMany(EmployeeLoan::class);
    }

    public function advances()
    {
        return $this->hasMany(EmployeeAdvance::class);
    }

    public function getFullNameAttribute()
    {
        return "{$this->first_name} {$this->last_name}";
    }

    public function getIsActiveAttribute()
    {
        return $this->status === 'active';
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeByDepartment($query, $departmentId)
    {
        return $query->where('department_id', $departmentId);
    }

    public function hasPayrollFor($month, $year)
    {
        return $this->payrolls()
            ->where('month', $month)
            ->where('year', $year)
            ->exists();
    }

    /**
     * Check if employee has complete KRA information
     */
    public function hasCompleteKraInfo()
    {
        return !empty($this->kra_pin) && !empty($this->nhif_number) && !empty($this->nssf_number);
    }

    /**
     * Get effective basic salary (from employee record or active contract)
     */
    public function getEffectiveBasicSalaryAttribute()
    {
        return $this->basic_salary ?? $this->activeContract?->salary ?? 0;
    }

    /**
     * Check if employee is eligible for payroll generation
     */
    public function isEligibleForPayroll()
    {
        return $this->status === 'active' && $this->effective_basic_salary > 0;
    }

    /**
     * Get monthly HELB deduction amount
     */
    public function getMonthlyHelbDeduction()
    {
        return $this->has_helb_loan ? $this->helb_monthly_deduction : 0;
    }
}
