<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\Builder;
use App\Traits\HasTenantScope;

class AuditLog extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'user_id',
        'organization_id',
        'event_type',
        'auditable_type',
        'auditable_id',
        'description',
        'ip_address',
        'user_agent',
        'url',
        'method',
        'old_values',
        'new_values',
        'metadata',
        'severity',
        'category',
    ];

    protected $casts = [
        'old_values' => 'array',
        'new_values' => 'array',
        'metadata' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the user who performed the action
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the organization this audit log belongs to
     */
    public function organization(): BelongsTo
    {
        return $this->belongsTo(Organization::class);
    }

    /**
     * Get the auditable model (polymorphic relationship)
     */
    public function auditable(): MorphTo
    {
        return $this->morphTo();
    }

    /**
     * Scope for filtering by organization
     */
    public function scopeForOrganization(Builder $query, ?int $organizationId): Builder
    {
        if ($organizationId === null) {
            return $query->whereNull('organization_id');
        }
        
        return $query->where('organization_id', $organizationId);
    }

    /**
     * Scope for filtering by user
     */
    public function scopeByUser(Builder $query, int $userId): Builder
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Scope for filtering by event type
     */
    public function scopeByEventType(Builder $query, string $eventType): Builder
    {
        return $query->where('event_type', $eventType);
    }

    /**
     * Scope for filtering by category
     */
    public function scopeByCategory(Builder $query, string $category): Builder
    {
        return $query->where('category', $category);
    }

    /**
     * Scope for filtering by date range
     */
    public function scopeDateRange(Builder $query, string $from, string $to): Builder
    {
        return $query->whereBetween('created_at', [$from, $to]);
    }

    /**
     * Scope for searching in descriptions
     */
    public function scopeSearch(Builder $query, string $search): Builder
    {
        return $query->where('description', 'like', "%{$search}%");
    }

    /**
     * Get formatted user name for display
     */
    public function getUserNameAttribute(): string
    {
        if (!$this->user) {
            return 'System';
        }

        return $this->user->name ?? 'Unknown User';
    }

    /**
     * Get formatted model name for display
     */
    public function getModelNameAttribute(): string
    {
        if (!$this->auditable_type) {
            return 'N/A';
        }

        return class_basename($this->auditable_type);
    }

    /**
     * Get severity badge color
     */
    public function getSeverityColorAttribute(): string
    {
        return match($this->severity) {
            'low' => 'success',
            'medium' => 'warning',
            'high' => 'danger',
            'critical' => 'dark',
            default => 'secondary'
        };
    }

    /**
     * Get category badge color
     */
    public function getCategoryColorAttribute(): string
    {
        return match($this->category) {
            'authentication' => 'primary',
            'authorization' => 'info',
            'data_access' => 'success',
            'data_modification' => 'warning',
            'system_admin' => 'danger',
            'compliance' => 'dark',
            'security' => 'danger',
            default => 'secondary'
        };
    }
}