<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;
use Carbon\Carbon;

class Attendance extends Model
{
    use HasFactory, HasTenantScope;

    protected $fillable = [
        'employee_id',
        'date',
        'check_in',
        'check_out',
        'worked_hours',
        'regular_hours',
        'break_hours',
        'paid_break_hours',
        'overtime_hours',
        'overtime_1_5x_hours',
        'overtime_2x_hours',
        'shift_type',
        'night_shift_hours',
        'weekend_hours',
        'holiday_hours',
        'included_in_payroll',
        'payroll_id',
        'payroll_period',
        'late_minutes',
        'early_departure_minutes',
        'deduction_hours',
        'approval_status',
        'approved_by',
        'approved_at',
        'status',
        'notes',
        'check_in_ip',
        'check_out_ip',
        'check_in_location',
        'check_out_location',
        'device_info',
        'organization_id',
    ];

    protected $casts = [
        'date' => 'date',
        'worked_hours' => 'decimal:2',
        'regular_hours' => 'decimal:2',
        'break_hours' => 'decimal:2',
        'paid_break_hours' => 'decimal:2',
        'overtime_hours' => 'decimal:2',
        'overtime_1_5x_hours' => 'decimal:2',
        'overtime_2x_hours' => 'decimal:2',
        'night_shift_hours' => 'decimal:2',
        'weekend_hours' => 'decimal:2',
        'holiday_hours' => 'decimal:2',
        'deduction_hours' => 'decimal:2',
        'late_minutes' => 'integer',
        'early_departure_minutes' => 'integer',
        'included_in_payroll' => 'boolean',
        'payroll_period' => 'date',
        'approved_at' => 'datetime',
    ];

    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    public function payroll()
    {
        return $this->belongsTo(Payroll::class);
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function calculateDetailedHours()
    {
        if (!$this->check_in || !$this->check_out) {
            return;
        }

        $checkIn = Carbon::parse($this->check_in);
        $checkOut = Carbon::parse($this->check_out);
        $totalMinutes = $checkOut->diffInMinutes($checkIn);
        
        // Get employee's contract for working hours
        $contract = $this->employee->activeContract;
        $standardHoursPerDay = $contract?->standard_hours_per_day ?? 8.0;
        $standardMinutesPerDay = $standardHoursPerDay * 60;
        
        // Calculate break time (assume 1 hour unpaid break for full day)
        $breakMinutes = $totalMinutes >= $standardMinutesPerDay ? 60 : 0;
        $paidBreakMinutes = min($breakMinutes, 30); // 30 minutes paid break
        
        // Net working minutes
        $netWorkingMinutes = $totalMinutes - $breakMinutes;
        $this->worked_hours = round($netWorkingMinutes / 60, 2);
        
        // Calculate regular vs overtime hours
        $regularMinutes = min($netWorkingMinutes, $standardMinutesPerDay);
        $this->regular_hours = round($regularMinutes / 60, 2);
        
        $overtimeMinutes = max(0, $netWorkingMinutes - $standardMinutesPerDay);
        $this->overtime_hours = round($overtimeMinutes / 60, 2);
        
        // Break down overtime (first 2 hours at 1.5x, rest at 2x)
        $overtime1_5xMinutes = min($overtimeMinutes, 120); // First 2 hours
        $overtime2xMinutes = max(0, $overtimeMinutes - 120); // Beyond 2 hours
        
        $this->overtime_1_5x_hours = round($overtime1_5xMinutes / 60, 2);
        $this->overtime_2x_hours = round($overtime2xMinutes / 60, 2);
        
        // Break hours
        $this->break_hours = round($breakMinutes / 60, 2);
        $this->paid_break_hours = round($paidBreakMinutes / 60, 2);
        
        // Determine shift type and premium hours
        $this->determineShiftType();
        
        // Calculate lateness and early departure
        $this->calculateLateness();
        
        $this->save();
    }

    protected function determineShiftType()
    {
        if (!$this->check_in) return;
        
        $checkIn = Carbon::parse($this->check_in);
        $dayOfWeek = $this->date->dayOfWeek;
        
        // Weekend work
        if (in_array($dayOfWeek, [Carbon::SATURDAY, Carbon::SUNDAY])) {
            $this->shift_type = 'weekend';
            $this->weekend_hours = $this->worked_hours;
            return;
        }
        
        // Holiday work (you might want to check against a holidays table)
        if ($this->isPublicHoliday()) {
            $this->shift_type = 'holiday';
            $this->holiday_hours = $this->worked_hours;
            return;
        }
        
        // Night shift (work starting after 6 PM or before 6 AM)
        $hour = $checkIn->hour;
        if ($hour >= 18 || $hour < 6) {
            $this->shift_type = 'night';
            $this->night_shift_hours = $this->worked_hours;
            return;
        }
        
        $this->shift_type = 'day';
    }

    protected function calculateLateness()
    {
        if (!$this->check_in || !$this->check_out) return;
        
        $contract = $this->employee->activeContract;
        if (!$contract) return;
        
        // Assume standard work start time is 8:00 AM
        $expectedStartTime = $this->date->copy()->setTime(8, 0);
        $expectedEndTime = $expectedStartTime->copy()->addHours($contract->standard_hours_per_day ?? 8);
        
        $checkIn = Carbon::parse($this->check_in);
        $checkOut = Carbon::parse($this->check_out);
        
        // Calculate lateness
        if ($checkIn->gt($expectedStartTime)) {
            $this->late_minutes = $checkIn->diffInMinutes($expectedStartTime);
        }
        
        // Calculate early departure
        if ($checkOut->lt($expectedEndTime)) {
            $this->early_departure_minutes = $expectedEndTime->diffInMinutes($checkOut);
        }
        
        // Calculate deduction hours (company policy dependent)
        $totalDeductionMinutes = $this->late_minutes + $this->early_departure_minutes;
        $this->deduction_hours = round($totalDeductionMinutes / 60, 2);
    }

    protected function isPublicHoliday()
    {
        // This is a simplified check - you might want to implement a proper holidays table
        $kenyanHolidays = [
            '01-01', // New Year
            '05-01', // Labour Day
            '06-01', // Madaraka Day
            '10-20', // Mashujaa Day
            '12-12', // Jamhuri Day
            '12-25', // Christmas
            '12-26', // Boxing Day
        ];
        
        $dateString = $this->date->format('m-d');
        return in_array($dateString, $kenyanHolidays);
    }

    public function getWorkedHoursFormattedAttribute()
    {
        $hours = floor($this->worked_hours);
        $minutes = ($this->worked_hours - $hours) * 60;
        return sprintf('%02d:%02d', $hours, $minutes);
    }

    public function getOvertimeHoursFormattedAttribute()
    {
        $hours = floor($this->overtime_hours);
        $minutes = ($this->overtime_hours - $hours) * 60;
        return sprintf('%02d:%02d', $hours, $minutes);
    }

    public function getTotalPayableHoursAttribute()
    {
        return $this->regular_hours + $this->overtime_hours - $this->deduction_hours;
    }

    public function calculatePayForContract(Contract $contract)
    {
        $regularPay = $this->regular_hours * $contract->effective_hourly_rate;
        $overtime1_5xPay = $this->overtime_1_5x_hours * ($contract->effective_hourly_rate * 1.5);
        $overtime2xPay = $this->overtime_2x_hours * ($contract->effective_hourly_rate * 2.0);
        
        // Premium pay for special shifts
        $nightShiftPremium = $this->night_shift_hours * ($contract->effective_hourly_rate * 0.25); // 25% premium
        $weekendPremium = $this->weekend_hours * ($contract->effective_hourly_rate * 0.5); // 50% premium
        $holidayPremium = $this->holiday_hours * ($contract->effective_hourly_rate * 1.0); // 100% premium
        
        // Deductions for lateness
        $latenessDeduction = $this->deduction_hours * $contract->effective_hourly_rate;
        
        return [
            'regular_pay' => $regularPay,
            'overtime_1_5x_pay' => $overtime1_5xPay,
            'overtime_2x_pay' => $overtime2xPay,
            'night_shift_premium' => $nightShiftPremium,
            'weekend_premium' => $weekendPremium,
            'holiday_premium' => $holidayPremium,
            'lateness_deduction' => $latenessDeduction,
            'total_pay' => $regularPay + $overtime1_5xPay + $overtime2xPay + 
                          $nightShiftPremium + $weekendPremium + $holidayPremium - $latenessDeduction,
        ];
    }

    public function scopeByMonth($query, $year, $month)
    {
        return $query->whereYear('date', $year)->whereMonth('date', $month);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    public function scopeNotInPayroll($query)
    {
        return $query->where('included_in_payroll', false);
    }

    public function scopeApproved($query)
    {
        return $query->where('approval_status', 'approved');
    }

    public function scopeByShiftType($query, $shiftType)
    {
        return $query->where('shift_type', $shiftType);
    }

    public function markAsIncludedInPayroll(Payroll $payroll)
    {
        $this->update([
            'included_in_payroll' => true,
            'payroll_id' => $payroll->id,
            'payroll_period' => $payroll->year . '-' . str_pad($payroll->month, 2, '0', STR_PAD_LEFT) . '-01',
        ]);
    }
}
