<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\HasTenantScope;

class ApplicationMessage extends Model
{
    use HasTenantScope;
    
    protected $fillable = [
        'application_id',
        'organization_id',
        'sender_type',
        'sender_id',
        'message',
        'read_by_candidate',
        'read_by_hr',
        'read_at',
    ];

    protected $casts = [
        'read_by_candidate' => 'boolean',
        'read_by_hr' => 'boolean',
        'read_at' => 'datetime',
    ];

    /**
     * Get the application that owns the message
     */
    public function application()
    {
        return $this->belongsTo(Application::class);
    }

    /**
     * Get the sender (HR user)
     */
    public function sender()
    {
        return $this->belongsTo(User::class, 'sender_id');
    }

    /**
     * Check if message is from HR
     */
    public function isFromHR()
    {
        return $this->sender_type === 'hr';
    }

    /**
     * Check if message is from candidate
     */
    public function isFromCandidate()
    {
        return $this->sender_type === 'candidate';
    }

    /**
     * Mark as read by candidate
     */
    public function markAsReadByCandidate()
    {
        if (!$this->read_by_candidate) {
            $this->update([
                'read_by_candidate' => true,
                'read_at' => now(),
            ]);
        }
    }

    /**
     * Mark as read by HR
     */
    public function markAsReadByHR()
    {
        if (!$this->read_by_hr) {
            $this->update([
                'read_by_hr' => true,
                'read_at' => now(),
            ]);
        }
    }
}
